param(
  [Parameter(Mandatory = $true)][string]$In,
  [Parameter(Mandatory = $true)][string]$Out
)

Set-StrictMode -Version Latest
$ErrorActionPreference = "Stop"

try { [Console]::InputEncoding = [System.Text.Encoding]::UTF8 } catch { }
try { [Console]::OutputEncoding = [System.Text.Encoding]::UTF8 } catch { }

# Ensure System.Net.Http types are available in Windows PowerShell.
try { Add-Type -AssemblyName System.Net.Http | Out-Null } catch { }

function WriteUtf8NoBom([string]$path, [string]$text) {
  $dir = [System.IO.Path]::GetDirectoryName([System.IO.Path]::GetFullPath($path))
  if ($dir -and -not (Test-Path -LiteralPath $dir)) { New-Item -ItemType Directory -Force -Path $dir | Out-Null }
  $utf8NoBom = New-Object System.Text.UTF8Encoding($false)
  [System.IO.File]::WriteAllText($path, $text, $utf8NoBom)
}

function ReadUtf8NoBom([string]$path) {
  $utf8NoBom = New-Object System.Text.UTF8Encoding($false)
  return [System.IO.File]::ReadAllText($path, $utf8NoBom)
}

function CN([int[]]$codepoints) {
  if ($null -eq $codepoints -or $codepoints.Count -eq 0) { return "" }
  $sb = New-Object System.Text.StringBuilder
  foreach ($cp in $codepoints) { [void]$sb.Append([char][int]$cp) }
  return $sb.ToString()
}

function Clamp([double]$x, [double]$lo, [double]$hi) {
  if ($x -lt $lo) { return $lo }
  if ($x -gt $hi) { return $hi }
  return $x
}

function TryGetDoubleProp($obj, [string[]]$names, [ref]$outVal) {
  if ($null -eq $obj -or $null -eq $names) { return $false }
  foreach ($n in $names) {
    if ([string]::IsNullOrWhiteSpace($n)) { continue }
    try {
      $p = $obj.PSObject.Properties[$n]
      if ($p -and $null -ne $p.Value) {
        $v = [double]$p.Value
        if (-not [double]::IsNaN($v) -and -not [double]::IsInfinity($v)) {
          $outVal.Value = $v
          return $true
        }
      }
    } catch { }
  }
  return $false
}

function SanitizeFileStem([string]$stem) {
  if ([string]::IsNullOrWhiteSpace($stem)) { return "curve" }
  $s = $stem.Trim()
  $s = [regex]::Replace($s, '[\\\\/:*?\"<>|]+', '_')
  $s = [regex]::Replace($s, '\s+', '_')
  if ($s.Length -gt 80) { $s = $s.Substring(0, 80) }
  if ([string]::IsNullOrWhiteSpace($s)) { return "curve" }
  return $s
}

function EnsureUniquePath([string]$path) {
  if ([string]::IsNullOrWhiteSpace($path)) { return $path }
  if (-not (Test-Path -LiteralPath $path)) { return $path }

  $dir = [System.IO.Path]::GetDirectoryName($path)
  $stem = [System.IO.Path]::GetFileNameWithoutExtension($path)
  $ext = [System.IO.Path]::GetExtension($path)
  for ($i = 1; $i -le 999; $i++) {
    $cand = Join-Path -Path $dir -ChildPath ("{0}_{1}{2}" -f $stem, $i, $ext)
    if (-not (Test-Path -LiteralPath $cand)) { return $cand }
  }
  return $path
}

function WriteRasCurveFromKeys([string]$outPath, [string]$name, [object[]]$keys) {
  if ($null -eq $keys -or $keys.Count -lt 2) { throw "not enough keys" }

  $lines = New-Object System.Collections.Generic.List[string]
  $lines.Add("name=$name")
  $lines.Add("type=rascurve")
  $lines.Add("RAS_SPEEDCURVE 1")

  $sorted = $keys | Sort-Object t
  $inv = [System.Globalization.CultureInfo]::InvariantCulture
  foreach ($k in $sorted) {
    $t = Clamp ([double]$k.t) 0.0 1.0
    $v = Clamp ([double]$k.v) 0.0 1.0
   $s = 0.0
   try { $s = [double]$k.s } catch { $s = 0.0 }
    # Format: t value inSlope outSlope inWeight outWeight mode
    $lines.Add(("{0} {1} {2} {3} {4} {5} {6}" -f $t.ToString("G17",$inv), $v.ToString("G17",$inv), $s.ToString("G17",$inv), $s.ToString("G17",$inv), (1.0/3.0).ToString("G17",$inv), (1.0/3.0).ToString("G17",$inv), 1))
  }

  WriteUtf8NoBom $outPath (($lines -join "`r`n") + "`r`n")
  return $outPath
}

function BuildRasCurveTextFromKeys([string]$name, [object[]]$keys) {
  if ($null -eq $keys -or $keys.Count -lt 2) { throw "not enough keys" }

  $lines = New-Object System.Collections.Generic.List[string]
  $lines.Add("RAS_SPEEDCURVE_TEMPLATE 1")
  $lines.Add("name=$name")
  $lines.Add("RAS_SPEEDCURVE 1")
  $lines.Add("# t value inSlope outSlope inWeight outWeight mode")

  $wgt = 1.0 / 3.0
  foreach ($k in ($keys | Sort-Object t)) {
    $t = Clamp ([double]$k.t) 0.0 1.0
    $v = Clamp ([double]$k.v) 0.0 1.0
    $s = 0.0
    try { $s = [double]$k.s } catch { $s = 0.0 }
    $inW = $wgt
    $outW = $wgt
    try { if ($k.PSObject.Properties["inWeight"]) { $inW = Clamp ([double]$k.inWeight) 0.0 1.0 } } catch { }
    try { if ($k.PSObject.Properties["outWeight"]) { $outW = Clamp ([double]$k.outWeight) 0.0 1.0 } } catch { }
    $modeK = 1
    try { if ($k.PSObject.Properties["mode"]) { $modeK = [int]$k.mode } } catch { $modeK = 1 }
    $inv = [System.Globalization.CultureInfo]::InvariantCulture
    $lines.Add(("{0} {1} {2} {2} {3} {4} {5}" -f $t.ToString("G17",$inv), $v.ToString("G17",$inv), $s.ToString("G17",$inv), $inW.ToString("G17",$inv), $outW.ToString("G17",$inv), $modeK))
  }
  return (($lines -join "`r`n") + "`r`n")
}

function MakeHardSpringFrontloadedKeys([double]$durationSec, [double]$centerValue, [double]$firstPeak, [double]$decay, [double]$bounces, [double]$tail, [double]$stiffness) {
  # Simple deterministic "hard spring" preset: a strong early overshoot then a quick settle.
  $keys = New-Object System.Collections.Generic.List[object]
  $keys.Add([pscustomobject]@{ t = 0.0; v = $centerValue; s = 0.0 })

  $peakT = Clamp (0.06 * (4.0 / [Math]::Max(0.5, $durationSec))) 0.03 0.12
  $keys.Add([pscustomobject]@{ t = $peakT; v = Clamp $firstPeak 0.0 1.0; s = 0.0 })

  $n = [int][Math]::Max(1, [Math]::Round($bounces))
  for ($i = 1; $i -le $n; $i++) {
    $u = $i / [double]($n + 1)
    $tt = Clamp ($peakT + (0.75 * (1.0 - $peakT)) * $u) 0.0 1.0
    $amp = (1.0 - $u)
    $amp = [Math]::Pow([Math]::Max(0.0, $amp), [Math]::Max(0.5, $stiffness))
    $amp *= [Math]::Max(0.0, $decay)
    $sign = if (($i % 2) -eq 1) { -1.0 } else { 1.0 }
    $vv = Clamp ($centerValue + $sign * $amp * (Clamp ($firstPeak - $centerValue) -1.0 1.0)) 0.0 1.0
    $keys.Add([pscustomobject]@{ t = $tt; v = $vv; s = 0.0 })
  }

  $keys.Add([pscustomobject]@{ t = Clamp (1.0 - [Math]::Max(0.0, $tail)) 0.0 1.0; v = $centerValue; s = 0.0 })
  $keys.Add([pscustomobject]@{ t = 1.0; v = $centerValue; s = 0.0 })
  return $keys.ToArray()
}

function MakeHeartbeatKeys([double]$durationSec, [double]$bpm) {
  $bpm = [Math]::Max(30.0, [Math]::Min(200.0, $bpm))
  $period = 60.0 / $bpm
  $beats = [int][Math]::Round($durationSec / $period)
  if ($beats -lt 1) { $beats = 1 }

  $template = @(
    @{ dt = 0.00; v = 0.00 },
    @{ dt = 0.04 * $period; v = 0.08 },  # small pre-bump
    @{ dt = 0.07 * $period; v = 1.00 },  # main spike (QRS)
    @{ dt = 0.10 * $period; v = 0.18 },  # fast fall
    @{ dt = 0.16 * $period; v = 0.00 },  # back to baseline
    @{ dt = 0.28 * $period; v = 0.35 },  # secondary bump (T wave)
    @{ dt = 0.38 * $period; v = 0.00 },  # settle
    @{ dt = $period; v = 0.00 }
  )

  $keys = New-Object System.Collections.Generic.List[object]
  for ($b = 0; $b -lt $beats; $b++) {
    $t0 = $b * $period
    foreach ($k in $template) {
      $tSec = $t0 + [double]$k.dt
      if ($tSec -lt -1e-9 -or $tSec -gt $durationSec + 1e-9) { continue }
      $keys.Add([pscustomobject]@{
        t = (Clamp ($tSec / $durationSec) 0.0 1.0)
        v = (Clamp ([double]$k.v) 0.0 1.0)
        s = 0.0
      })
    }
  }
  $keys.Add([pscustomobject]@{ t = 1.0; v = 0.15; s = 0.0 })

  $ordered = $keys | Sort-Object t
  $dedup = New-Object System.Collections.Generic.List[object]
  foreach ($k in $ordered) {
    if ($dedup.Count -eq 0 -or [Math]::Abs([double]$k.t - [double]$dedup[$dedup.Count - 1].t) -gt 1e-6) {
      $dedup.Add($k)
    } else {
      $dedup[$dedup.Count - 1] = $k
    }
  }
  return $dedup.ToArray()
}

function TryExtractBpmFromPrompt([string]$prompt, [double]$defaultBpm) {
  $defaultBpm = [Math]::Max(30.0, [Math]::Min(200.0, $defaultBpm))
  if ([string]::IsNullOrWhiteSpace($prompt)) { return $defaultBpm }
  $p = $prompt
  $m = [regex]::Match($p, '(?i)(\\d{2,3}(?:\\.\\d+)?)\\s*bpm')
  if ($m.Success) {
    $v = 0.0
    if ([double]::TryParse($m.Groups[1].Value, [ref]$v)) { return [Math]::Max(30.0, [Math]::Min(200.0, $v)) }
  }
  # Also accept "BPM: 75" or "75 BPM" and a lone number if the prompt is very short and pulse-like.
  $m2 = [regex]::Match($p, '(?i)bpm\\s*[:=\\-]?\\s*(\\d{2,3}(?:\\.\\d+)?)')
  if ($m2.Success) {
    $v2 = 0.0
    if ([double]::TryParse($m2.Groups[1].Value, [ref]$v2)) { return [Math]::Max(30.0, [Math]::Min(200.0, $v2)) }
  }
  if ((PromptHasPulseIntent $prompt) -and ($p -match '^(\\s*\\d{2,3}(?:\\.\\d+)?\\s*)$')) {
    $v3 = 0.0
    if ([double]::TryParse($Matches[1], [ref]$v3)) { return [Math]::Max(30.0, [Math]::Min(200.0, $v3)) }
  }
  return $defaultBpm
}

function InvokeAiPipeline([string]$prompt, [string]$outRasPath, [string]$model, [string]$baseUrl, [string]$apiKey, [bool]$useOnline) {
  $rootDir = [System.IO.Path]::GetFullPath((Join-Path -Path $PSScriptRoot -ChildPath ".."))
  $pluginDir = Join-Path -Path $rootDir -ChildPath (CN @(0x7280,0x725B,0x52A8,0x753B,0x63D2,0x4EF6))
  $aiDir = Join-Path -Path $pluginDir -ChildPath "AI"
  $toolTextToDsl = Join-Path -Path $aiDir -ChildPath "text_to_dsl.ps1"
  $toolDslToRas = Join-Path -Path $aiDir -ChildPath "dsl_to_rascurve.ps1"
  if (-not (Test-Path -LiteralPath $toolTextToDsl)) { throw "Missing tool: $toolTextToDsl" }
  if (-not (Test-Path -LiteralPath $toolDslToRas)) { throw "Missing tool: $toolDslToRas" }

  $outRasPath = EnsureUniquePath $outRasPath
  $dslPath = [System.IO.Path]::ChangeExtension($outRasPath, ".dsl.json")

  if ($useOnline) {
    & $toolTextToDsl -Prompt $prompt -Out $dslPath -BaseUrl $baseUrl -Model $model -ApiKey $apiKey -Temperature 0.0 | Out-Null
  } else {
    & $toolTextToDsl -Prompt $prompt -Out $dslPath -LocalOnly | Out-Null
  }
  if (-not (Test-Path -LiteralPath $dslPath)) { throw "Failed to generate DSL." }

  & $toolDslToRas -In $dslPath -Out $outRasPath | Out-Null
  if (-not (Test-Path -LiteralPath $outRasPath)) { throw "Failed to generate rascurve." }

  return $outRasPath
}

$script:tlsInit = $false
function EnsureTls12() {
  if ($script:tlsInit) { return }
  $script:tlsInit = $true
  try { [Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12 } catch { }
}

function InvokeHttpPostJson([string]$url, $payload, [int]$timeoutSec) {
  EnsureTls12
  if ([string]::IsNullOrWhiteSpace($url)) { throw "Missing url." }
  $timeoutSec = [Math]::Max(5, [Math]::Min(600, $timeoutSec))

  $handler = New-Object System.Net.Http.HttpClientHandler
  try { $handler.AutomaticDecompression = [System.Net.DecompressionMethods]::GZip -bor [System.Net.DecompressionMethods]::Deflate } catch { }

  $client = New-Object System.Net.Http.HttpClient($handler)
  $client.Timeout = [TimeSpan]::FromSeconds($timeoutSec)

  $json = ($payload | ConvertTo-Json -Depth 12 -Compress)
  $content = New-Object System.Net.Http.StringContent($json, [System.Text.Encoding]::UTF8, "application/json")
  $resp = $client.PostAsync($url, $content).GetAwaiter().GetResult()
  $text = $resp.Content.ReadAsStringAsync().GetAwaiter().GetResult()
  if (-not $resp.IsSuccessStatusCode) {
    throw ("HTTP {0}: {1}" -f ([int]$resp.StatusCode), ($text | Out-String).Trim())
  }
  if ([string]::IsNullOrWhiteSpace($text)) { throw "Empty response." }
  return ($text | ConvertFrom-Json)
}

function InvokeRhinobirdServerCurve([string]$serverUrl, [string]$username, [string]$deviceId, [string]$prompt, [string]$modify, [double]$durationSec, [int]$variantsCount, $curveTv) {
  if ([string]::IsNullOrWhiteSpace($serverUrl)) { throw "Missing serverUrl." }
  if ([string]::IsNullOrWhiteSpace($username)) { throw "Missing username." }
  if ([string]::IsNullOrWhiteSpace($deviceId)) { throw "Missing deviceId." }
  if ([string]::IsNullOrWhiteSpace($prompt)) { throw "Prompt is empty." }

  $base = $serverUrl.TrimEnd("/")
  $path = if ([string]::IsNullOrWhiteSpace($modify)) { "/api/ai/curve/generate" } else { "/api/ai/curve/revise" }
  $url = $base + $path

  $body = [ordered]@{
    username = $username
    device_id = $deviceId
    prompt = $prompt
    duration_sec = [double]$durationSec
    variants = [int]([Math]::Max(1, [Math]::Min(3, $variantsCount)))
    current_curve = @()
  }
  if (-not [string]::IsNullOrWhiteSpace($modify)) { $body.modify = $modify }
  if ($null -ne $curveTv -and ($curveTv -is [System.Array]) -and $curveTv.Count -ge 2) { $body.current_curve = $curveTv }

  return (InvokeHttpPostJson $url $body 240)
}

function MaybeFixUtf8Mojibake([string]$s) {
  if ([string]::IsNullOrWhiteSpace($s)) { return $s }
  $orig = $s
  $origCjk = 0
  try { $origCjk = [regex]::Matches($orig, '[\u4e00-\u9fff]').Count } catch { $origCjk = 0 }
  # NOTE: Some OpenAI-compatible gateways return UTF-8 bytes but label the response as GBK/CP936.
  # That produces "CJK-looking" mojibake (with occasional PUA/Euro chars), so we can't early-return
  # just because the string contains CJK characters.

  $hasLatin1Hi = $false
  try { $hasLatin1Hi = ($orig -match '[\u00c2-\u00ff]') } catch { $hasLatin1Hi = $false }
  $hasWeird = $false
  try {
    # U+E000..U+F8FF is Private Use Area; U+20AC is Euro; both often appear in this mojibake pattern.
    $hasWeird = ($orig -match '[\uE000-\uF8FF\u20AC\uFFFD]')
  } catch { $hasWeird = $false }
  if (-not $hasLatin1Hi -and -not $hasWeird) { return $orig }

  $best = $orig
  $bestScore = -1

  function ScoreText([string]$txt) {
    if ([string]::IsNullOrWhiteSpace($txt)) { return -1 }
    $cjk = 0
    $weird2 = 0
    try { $cjk = [regex]::Matches($txt, '[\u4e00-\u9fff]').Count } catch { $cjk = 0 }
    try { $weird2 = [regex]::Matches($txt, '[\uE000-\uF8FF\u20AC\uFFFD]').Count } catch { $weird2 = 0 }
    # Prefer more CJK, penalize weird chars heavily.
    return ($cjk * 10) - ($weird2 * 30)
  }

  $bestScore = (ScoreText $best)
  try {
    $latin1 = [System.Text.Encoding]::GetEncoding(28591)
    $bytes = $latin1.GetBytes($orig)
    $fixed = [System.Text.Encoding]::UTF8.GetString($bytes)
    $sc = (ScoreText $fixed)
    if ($sc -gt $bestScore) { $best = $fixed; $bestScore = $sc }
  } catch { }

  # Try reversing a GBK/CP936 mis-decode: encode as CP936 bytes, decode as UTF-8.
  try {
    $gbk = [System.Text.Encoding]::GetEncoding(936)
    $bytes2 = $gbk.GetBytes($orig)
    $fixed2 = [System.Text.Encoding]::UTF8.GetString($bytes2)
    $sc2 = (ScoreText $fixed2)
    if ($sc2 -gt $bestScore) { $best = $fixed2; $bestScore = $sc2 }
  } catch { }

  return $best
}

function ExtractChatContentText($respObj) {
  if ($null -eq $respObj) { return "" }
  $c = $null
  try { $c = $respObj.choices[0].message.content } catch { $c = $null }
  if ($null -eq $c) {
    try { $c = $respObj.choices[0].text } catch { $c = $null }
  }
  if ($null -eq $c) { return "" }
  $t = ""
  if ($c -is [System.Array]) {
    $parts = New-Object System.Collections.Generic.List[string]
    foreach ($p in $c) {
      if ($null -eq $p) { continue }
      try {
        $tp = $p.PSObject.Properties["text"]
        if ($tp -and $tp.Value) { $parts.Add([string]$tp.Value); continue }
      } catch { }
      try {
        $cp = $p.PSObject.Properties["content"]
        if ($cp -and $cp.Value) { $parts.Add([string]$cp.Value); continue }
      } catch { }
      $parts.Add([string]$p)
    }
    $t = ($parts -join "")
  } else {
    try {
      $tp = $c.PSObject.Properties["text"]
      if ($tp -and $tp.Value) { $t = [string]$tp.Value }
    } catch { }
    if ([string]::IsNullOrWhiteSpace($t)) {
      try {
        $cp = $c.PSObject.Properties["content"]
        if ($cp -and $cp.Value) { $t = [string]$cp.Value }
      } catch { }
    }
    if ([string]::IsNullOrWhiteSpace($t)) { $t = [string]$c }
  }
  return (MaybeFixUtf8Mojibake $t).Trim()
}

function InvokeChatCompletions([string]$baseUrl, [string]$model, [string]$apiKey, [string]$system, [string]$user, [double]$temperature, [int]$maxTokens) {
  EnsureTls12
  $base = $baseUrl.TrimEnd("/")
  $uri = "$base/chat/completions"

  $body = @{
    model = $model
    temperature = [Math]::Max(0.0, [Math]::Min(2.0, $temperature))
    max_tokens = [Math]::Max(64, $maxTokens)
    messages = @(
      @{ role = "system"; content = $system }
      @{ role = "user"; content = $user }
    )
  }

  # Use HttpClient for a hard timeout (Invoke-RestMethod can hang on some network edge cases).
  $json = $body | ConvertTo-Json -Depth 10
  $handler = $null
  $client = $null
  $req = $null
  try {
    $handler = New-Object System.Net.Http.HttpClientHandler
    try { $handler.AutomaticDecompression = [System.Net.DecompressionMethods]::GZip -bor [System.Net.DecompressionMethods]::Deflate } catch { }

    $client = New-Object System.Net.Http.HttpClient($handler)
    # Some upstreams (especially local models) can take a while on long prompts/variants.
    $client.Timeout = [TimeSpan]::FromSeconds(240)

    $req = New-Object System.Net.Http.HttpRequestMessage([System.Net.Http.HttpMethod]::Post, $uri)
    $req.Headers.Authorization = New-Object System.Net.Http.Headers.AuthenticationHeaderValue("Bearer", $apiKey)
    $req.Content = New-Object System.Net.Http.StringContent($json, [System.Text.Encoding]::UTF8, "application/json")

    $resp = $client.SendAsync($req).GetAwaiter().GetResult()
    $respText = $resp.Content.ReadAsStringAsync().GetAwaiter().GetResult()
    if (-not $resp.IsSuccessStatusCode) {
      $code = [int]$resp.StatusCode
      $reason = [string]$resp.ReasonPhrase
      $snippet = $respText
      if ($null -eq $snippet) { $snippet = "" }
      $snippet = $snippet.Trim()
      if ($snippet.Length -gt 1200) { $snippet = $snippet.Substring(0, 1200) + "..." }
      throw ("HTTP {0} {1}. Response body:`n{2}" -f $code, $reason, $snippet)
    }
    return ($respText | ConvertFrom-Json)
  } catch {
    $msg = $_.Exception.Message
    if ([string]::IsNullOrWhiteSpace($msg)) { $msg = $_.ToString() }
    throw $msg
  } finally {
    try { if ($null -ne $req) { $req.Dispose() } } catch { }
    try { if ($null -ne $client) { $client.Dispose() } } catch { }
    try { if ($null -ne $handler) { $handler.Dispose() } } catch { }
  }
}

function StripCodeFences([string]$s) {
  if ($null -eq $s) { return "" }
  $t = $s.Trim()
  $i0 = $t.IndexOf('```')
  if ($i0 -ge 0) {
    $i1 = $t.IndexOf('```', $i0 + 3)
    if ($i1 -gt $i0) {
      $inner = $t.Substring($i0 + 3, $i1 - ($i0 + 3))
      return $inner.Trim()
    }
  }
  return $t
}

function ExtractBalancedJsonObject([string]$s) {
  if ([string]::IsNullOrWhiteSpace($s)) { return $null }
  $t = StripCodeFences $s
  $iStart = $t.IndexOf('{')
  if ($iStart -lt 0) { return $null }

  $depth = 0
  $inString = $false
  $escape = $false
  for ($i = $iStart; $i -lt $t.Length; $i++) {
    $ch = $t[$i]
    if ($inString) {
      if ($escape) { $escape = $false; continue }
      if ($ch -eq [char]92) { $escape = $true; continue } # '\'
      if ($ch -eq '"') { $inString = $false; continue }
      continue
    }

    if ($ch -eq '"') { $inString = $true; continue }
    if ($ch -eq '{') { $depth++; continue }
    if ($ch -eq '}') {
      $depth--
      if ($depth -eq 0) {
        try { return $t.Substring($iStart, ($i - $iStart + 1)) } catch { return $null }
      }
      continue
    }
  }
  return $null
}

function ExtractBalancedJsonArray([string]$s) {
  if ([string]::IsNullOrWhiteSpace($s)) { return $null }
  $t = StripCodeFences $s
  $iStart = $t.IndexOf('[')
  if ($iStart -lt 0) { return $null }

  $depth = 0
  $inString = $false
  $escape = $false
  for ($i = $iStart; $i -lt $t.Length; $i++) {
    $ch = $t[$i]
    if ($inString) {
      if ($escape) { $escape = $false; continue }
      if ($ch -eq [char]92) { $escape = $true; continue } # '\'
      if ($ch -eq '"') { $inString = $false; continue }
      continue
    }

    if ($ch -eq '"') { $inString = $true; continue }
    if ($ch -eq '[') { $depth++; continue }
    if ($ch -eq ']') {
      $depth--
      if ($depth -eq 0) {
        try { return $t.Substring($iStart, ($i - $iStart + 1)) } catch { return $null }
      }
      continue
    }
  }
  return $null
}

function ParseJsonFromModelText([string]$text, [bool]$requireObject = $true) {
  $raw = StripCodeFences $text
  $raw = $raw.Trim()
  if ([string]::IsNullOrWhiteSpace($raw)) { throw "Model returned empty text (expected JSON)." }

  # Fast path: the whole content is valid JSON (common for strict models).
  try {
    $obj0 = ($raw | ConvertFrom-Json)
    if ($requireObject -and ($obj0 -is [System.Array])) { throw "Model returned a JSON array (expected a JSON object)." }
    return $obj0
  } catch { }

  # Fallback: extract a balanced JSON object/array from mixed text (e.g. leading 'thinking' lines).
  $frag = ExtractBalancedJsonObject $raw
  if ([string]::IsNullOrWhiteSpace($frag)) { $frag = ExtractBalancedJsonArray $raw }
  if ([string]::IsNullOrWhiteSpace($frag)) {
    $snip = $raw
    if ($snip.Length -gt 420) { $snip = $snip.Substring(0, 420) + "..." }
    throw ("Model did not return JSON. Got: " + $snip)
  }

  try {
    $obj = ($frag | ConvertFrom-Json)
    if ($requireObject -and ($obj -is [System.Array])) { throw "Model returned a JSON array (expected a JSON object)." }
    return $obj
  } catch {
    $sn = $frag
    if ($sn.Length -gt 220) { $sn = $sn.Substring(0, 220) + "..." }
    throw ("Invalid JSON from model. raw=" + $sn)
  }
}

function HasProp($obj, [string]$name) {
  if ($null -eq $obj) { return $false }
  if ([string]::IsNullOrWhiteSpace($name)) { return $false }
  try { return ($null -ne $obj.PSObject.Properties[$name]) } catch { return $false }
}

function EnsureStackDslHasBase($dsl, [bool]$expectBounce) {
  if ($null -eq $dsl) { return $dsl }
  $m = ""
  try { $m = [string]$dsl.model } catch { $m = "" }
  if ([string]::IsNullOrWhiteSpace($m)) { return $dsl }
  if ($m.Trim().ToLowerInvariant() -ne "stack") { return $dsl }

  if (-not (HasProp $dsl "base") -or $null -eq $dsl.base) {
    $baseModel = if ($expectBounce) { "spring" } else { "ease" }
    try { $dsl | Add-Member -NotePropertyName "base" -NotePropertyValue ([pscustomobject]@{ model = $baseModel }) -Force } catch { }
  }

  $bm = ""
  try { $bm = [string]$dsl.base.model } catch { $bm = "" }
  if ([string]::IsNullOrWhiteSpace($bm)) {
    $baseModel2 = if ($expectBounce) { "spring" } else { "ease" }
    try { $dsl.base.model = $baseModel2 } catch { }
  }

  return $dsl
}

function TryGetSketchTvFromInputs($inputs) {
  try {
    if ($null -eq $inputs) { return @() }
    $sk = $inputs.PSObject.Properties["sketch"]
    if ($null -eq $sk -or $null -eq $sk.Value) { return @() }
    $tv = $sk.Value.PSObject.Properties["tv"]
    if ($null -eq $tv -or $null -eq $tv.Value) { return @() }
    # Expect array of [t,v] pairs.
    $pairs = New-Object System.Collections.Generic.List[object]
    foreach ($p in $tv.Value) {
      try {
        if ($p -is [System.Array] -and $p.Length -ge 2) {
          $t = [double]$p[0]
          $v = [double]$p[1]
          $pairs.Add([pscustomobject]@{ t = (Clamp $t 0.0 1.0); v = (Clamp $v 0.0 1.0) }) | Out-Null
        }
      } catch { }
    }
    return $pairs.ToArray()
  } catch {
    return @()
  }
}

function GetSketchModeFromInputs($inputs) {
  $m = ""
  try { $m = [string]$inputs.sketchMode } catch { $m = "" }
  if ([string]::IsNullOrWhiteSpace($m)) { return "constraint" }
  $m = $m.Trim().ToLowerInvariant()
  if ($m -ne "constraint" -and $m -ne "reference" -and $m -ne "none") { return "constraint" }
  return $m
}

function TryGetPrevCurveTvFromInputs($inputs) {
  try {
    $pc = $inputs.prevCurve
    if ($null -eq $pc) { return $null }
    $tv = $pc.tv
    if ($null -eq $tv -or -not ($tv -is [System.Array]) -or $tv.Count -lt 4) { return $null }
    $out = New-Object System.Collections.Generic.List[object]
    foreach ($pair in $tv) {
      try {
        $t = [double]$pair[0]
        $v = [double]$pair[1]
        $out.Add([pscustomobject]@{ t = (Clamp $t 0.0 1.0); v = (Clamp $v 0.0 1.0) }) | Out-Null
      } catch { }
    }
    if ($out.Count -lt 4) { return $null }
    return $out.ToArray()
  } catch { return $null }
}

function TryGetSelectedCurveTvFromInputs($inputs) {
  try {
    $pc = $inputs.selectedCurve
    if ($null -eq $pc) { return $null }
    $tv = $pc.tv
    if ($null -eq $tv -or -not ($tv -is [System.Array]) -or $tv.Count -lt 4) { return $null }
    $out = New-Object System.Collections.Generic.List[object]
    foreach ($pair in $tv) {
      try {
        $t = [double]$pair[0]
        $v = [double]$pair[1]
        $out.Add([pscustomobject]@{ t = (Clamp $t 0.0 1.0); v = (Clamp $v 0.0 1.0) }) | Out-Null
      } catch { }
    }
    if ($out.Count -lt 4) { return $null }
    return $out.ToArray()
  } catch { return $null }
}

function TryGetChatTextFromInputs($inputs) {
  try {
    $turns = $inputs.chatTurns
    if ($null -eq $turns) { return "" }
    if (-not ($turns -is [System.Array])) { return "" }
    return (FormatChatTurnsForAi $turns 10000)
  } catch { return "" }
}

function FormatSketchPairs($tv, [int]$maxPairs) {
  if ($null -eq $tv) { return "" }
  $maxPairs = [Math]::Max(8, [Math]::Min(128, $maxPairs))
  $inv = [System.Globalization.CultureInfo]::InvariantCulture
  $pairs = New-Object System.Collections.Generic.List[string]
  $n = 0
  foreach ($p in $tv) {
    if ($n -ge $maxPairs) { break }
    try {
      $pairs.Add(("{0},{1}" -f ([double]$p.t).ToString("G4",$inv), ([double]$p.v).ToString("G4",$inv))) | Out-Null
      $n++
    } catch { }
  }
  return ($pairs -join "; ")
}

function ParseRasCurveTV([string]$rasPath) {
  $text = Get-Content -Raw -Encoding UTF8 -LiteralPath $rasPath
  $lines = $text -split "(`r`n|`n|`r)"
  $keys = New-Object System.Collections.Generic.List[object]
  $inCurve = $false
  foreach ($raw in $lines) {
    $line = $raw.Trim()
    if ([string]::IsNullOrWhiteSpace($line)) { continue }
    if ($line.StartsWith("#")) { continue }
    if ($line -eq "RAS_SPEEDCURVE 1") { $inCurve = $true; continue }
    if (-not $inCurve) { continue }
    $parts = $line -split '\s+'
    if ($parts.Count -lt 2) { continue }
    try {
      $t = [double]$parts[0]
      $v = [double]$parts[1]
      $keys.Add([pscustomobject]@{ t = $t; v = $v })
    } catch { }
  }
  return $keys.ToArray()
}

function ParseRasCurveFullKeysFromText([string]$text) {
  if ($null -eq $text) { $text = "" }
  $lines = $text -split "(`r`n|`n|`r)"
  $keys = New-Object System.Collections.Generic.List[object]
  foreach ($raw in $lines) {
    $line = $raw
    if ($null -eq $line) { continue }
    $s = $line.Trim()
    if ([string]::IsNullOrWhiteSpace($s)) { continue }
    if ($s.StartsWith("#") -or $s.StartsWith(";")) { continue }
    if ($s.StartsWith("RAS_")) { continue }
    if ($s.StartsWith("name=") -or $s.StartsWith("type=")) { continue }

    $parts = $s -split "\\s+"
    if ($parts.Count -lt 2) { continue }
    $t = 0.0; $v = 0.0
    if (-not [double]::TryParse($parts[0], [ref]$t)) { continue }
    if (-not [double]::TryParse($parts[1], [ref]$v)) { continue }

    $inSlope = 0.0; $outSlope = 0.0
    $inW = 1.0/3.0; $outW = 1.0/3.0
    $mode = 1
    if ($parts.Count -ge 7) {
      [void][double]::TryParse($parts[2], [ref]$inSlope)
      [void][double]::TryParse($parts[3], [ref]$outSlope)
      [void][double]::TryParse($parts[4], [ref]$inW)
      [void][double]::TryParse($parts[5], [ref]$outW)
      [void][int]::TryParse($parts[6], [ref]$mode)
    }

    $keys.Add([pscustomobject]@{
      t = (Clamp $t 0.0 1.0)
      value = (Clamp $v 0.0 1.0)
      inSlope = $inSlope
      outSlope = $outSlope
      inWeight = (Clamp $inW 0.0 1.0)
      outWeight = (Clamp $outW 0.0 1.0)
      mode = $mode
    }) | Out-Null
  }
  return @($keys | Sort-Object t)
}

function Bezier1D([double]$p0, [double]$p1, [double]$p2, [double]$p3, [double]$s) {
  $inv = 1.0 - $s
  $inv2 = $inv * $inv
  $s2 = $s * $s
  return ($inv2 * $inv) * $p0 + (3.0 * $inv2 * $s) * $p1 + (3.0 * $inv * $s2) * $p2 + ($s2 * $s) * $p3
}

function BezierDeriv1D([double]$p0, [double]$p1, [double]$p2, [double]$p3, [double]$s) {
  $inv = 1.0 - $s
  return 3.0 * $inv * $inv * ($p1 - $p0) + 6.0 * $inv * $s * ($p2 - $p1) + 3.0 * $s * $s * ($p3 - $p2)
}

function EvaluateRasCurveAt($keys, [double]$time) {
  if ($null -eq $keys -or $keys.Count -lt 2) { return $time }
  if ($time -le [double]$keys[0].t) { return [double]$keys[0].value }
  if ($time -ge [double]$keys[$keys.Count - 1].t) { return [double]$keys[$keys.Count - 1].value }

  $prev = $keys[0]
  $next = $null
  for ($i = 1; $i -lt $keys.Count; $i++) {
    if ($time -le [double]$keys[$i].t) {
      $prev = $keys[$i - 1]
      $next = $keys[$i]
      break
    }
  }
  if ($null -eq $next) { return [double]$keys[$keys.Count - 1].value }

  $t0 = [double]$prev.t
  $t1 = [double]$next.t
  $dt = $t1 - $t0
  if ($dt -le 1e-12) { return [double]$prev.value }

  $w0 = [double]$prev.outWeight
  $w1 = [double]$next.inWeight
  $sum = $w0 + $w1
  if ($sum -gt 0.999) {
    $scale = 0.999 / $sum
    $w0 *= $scale
    $w1 *= $scale
  }

  $x0 = $t0; $x3 = $t1
  $x1 = $t0 + $w0 * $dt
  $x2 = $t1 - $w1 * $dt
  if ($x2 -lt $x1) {
    $mid = 0.5 * ($t0 + $t1)
    $x1 = $mid; $x2 = $mid
  }

  $y0 = [double]$prev.value
  $y3 = [double]$next.value
  $linearSlope = ($y3 - $y0) / $dt
  $m0 = [double]$prev.outSlope
  $m1 = [double]$next.inSlope
  $y1 = $y0 + $m0 * ($x1 - $x0)
  $y2 = $y3 - $m1 * ($x3 - $x2)

  $s = ($time - $x0) / [Math]::Max(1e-9, ($x3 - $x0))
  $s = Clamp $s 0.0 1.0
  $converged = $false
  for ($iter = 0; $iter -lt 8; $iter++) {
    $xs = Bezier1D $x0 $x1 $x2 $x3 $s
    $dx = BezierDeriv1D $x0 $x1 $x2 $x3 $s
    $diff = $xs - $time
    if ([Math]::Abs($diff) -le 1e-6) { $converged = $true; break }
    if ([Math]::Abs($dx) -lt 1e-12) { break }
    $ns = $s - ($diff / $dx)
    if ($ns -lt -0.25 -or $ns -gt 1.25) { break }
    $s = Clamp $ns 0.0 1.0
  }
  if (-not $converged) {
    $lo = 0.0; $hi = 1.0
    for ($iter = 0; $iter -lt 24; $iter++) {
      $mid = 0.5 * ($lo + $hi)
      $xs = Bezier1D $x0 $x1 $x2 $x3 $mid
      if ($xs -lt $time) { $lo = $mid } else { $hi = $mid }
    }
    $s = 0.5 * ($lo + $hi)
  }
  $val = Bezier1D $y0 $y1 $y2 $y3 $s
  return (Clamp $val 0.0 1.0)
}

function ComputeSketchFit($keysFull, $sketchTv) {
  if ($null -eq $keysFull -or $keysFull.Count -lt 2 -or $null -eq $sketchTv -or $sketchTv.Count -lt 8) {
    return [pscustomobject]@{ ok = $false; mae = 0.0; maxe = 0.0 }
  }
  $sum = 0.0
  $maxe = 0.0
  $n = 0
  foreach ($p in $sketchTv) {
    $t = [double]$p.t
    $v = [double]$p.v
    $y = EvaluateRasCurveAt $keysFull $t
    $e = [Math]::Abs($y - $v)
    $sum += $e
    if ($e -gt $maxe) { $maxe = $e }
    $n++
  }
  if ($n -le 0) { return [pscustomobject]@{ ok = $false; mae = 0.0; maxe = 0.0 } }
  return [pscustomobject]@{ ok = $true; mae = ($sum / $n); maxe = $maxe }
}

function ComputeStats($keys) {
  $minV = 0.0
  $maxV = 0.0
  $cnt = 0
  if ($null -ne $keys) { $cnt = $keys.Count }
  if ($cnt -gt 0) {
    $minV = [double]$keys[0].v
    $maxV = [double]$keys[0].v
    foreach ($k in $keys) {
      $v = [double]$k.v
      if ($v -lt $minV) { $minV = $v }
      if ($v -gt $maxV) { $maxV = $v }
    }
  }
  return [pscustomobject]@{ keyCount = $cnt; minV = $minV; maxV = $maxV }
}

function BuildAiSummaryInput([string]$modeName, [string]$userPrompt, [string]$outPath, [object[]]$keys, [object]$stats) {
  $file = Split-Path -Leaf $outPath
  $pairsText = ""
  try {
    $keyPairs = $keys | Select-Object -First 40 | ForEach-Object { ("{0},{1}" -f ([double]$_.t).ToString("G17"), ([double]$_.v).ToString("G17")) }
    $pairsText = ($keyPairs -join "; ")
  } catch { $pairsText = "" }
  $p = $userPrompt
  if ([string]::IsNullOrWhiteSpace($p)) { $p = "(none)" }

  return @"
Mode: $modeName
Output: $file
KeyCount: $($stats.keyCount)
ValueRange: [$([Math]::Round($stats.minV,6)), $([Math]::Round($stats.maxV,6))]
UserPrompt: $p
Keys(t,v) sample: $pairsText
"@
}

function GenerateAiSummary([string]$baseUrl, [string]$model, [string]$apiKey, [string]$modeName, [string]$userPrompt, [string]$outPath, [object[]]$keys, [object]$stats) {
  # IMPORTANT: the GUI generates a deterministic local analysis. Ask the model ONLY for improvement suggestions,
  # so it won't hallucinate a curve style that doesn't match the actual preview.
  $system = "You are an animation curve assistant. Output Chinese plain text ONLY. Task: propose 8-12 prompt keywords (each with a short explanation) to modify the NEXT generation. Base your suggestions on the provided key samples (actual curve) + the user's intent. Do NOT restate what curve it is, do NOT claim it is a spring/bounce unless the keys clearly show oscillation. Focus on actionable keywords like: more bouncy, fewer bounces, stiffer, softer, earlier peak, later peak, settle faster, add overshoot, remove recoil, etc."
  $user = BuildAiSummaryInput $modeName $userPrompt $outPath $keys $stats
  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user 0.3 500
  $txt = ExtractChatContentText $resp
  return $txt.Trim()
}

function AnalyzeKeysBasic($keys) {
  if ($null -eq $keys -or $keys.Count -lt 2) {
    return [pscustomobject]@{ ok = $false; dirChanges = 0; monotonicInc = $false; monotonicDec = $false }
  }

  $eps = 1e-6
  $prevSign = 0
  $dirChanges = 0
  $monInc = $true
  $monDec = $true

  for ($i = 1; $i -lt $keys.Count; $i++) {
    $dv = ([double]$keys[$i].v) - ([double]$keys[$i - 1].v)
    if ($dv -gt $eps) { $monDec = $false }
    if ($dv -lt (-$eps)) { $monInc = $false }

    $s = 0
    if ($dv -gt $eps) { $s = 1 }
    elseif ($dv -lt (-$eps)) { $s = -1 }
    if ($s -ne 0) {
      if ($prevSign -ne 0 -and $s -ne $prevSign) { $dirChanges++ }
      $prevSign = $s
    }
  }

  return [pscustomobject]@{ ok = $true; dirChanges = $dirChanges; monotonicInc = $monInc; monotonicDec = $monDec }
}

function SampleCurveValues($keysFull, [int]$n) {
  $n = [Math]::Max(32, [Math]::Min(512, $n))
  if ($null -eq $keysFull -or $keysFull.Count -lt 2) { return @() }
  $arr = New-Object double[] $n
  for ($i = 0; $i -lt $n; $i++) {
    $t = $i / [double]($n - 1)
    $arr[$i] = [double](EvaluateRasCurveAt $keysFull $t)
  }
  return $arr
}

function AnalyzeSampled($vals) {
  if ($null -eq $vals -or $vals.Length -lt 3) {
    return [pscustomobject]@{ ok = $false; dirChanges = 0; monotonicInc = $false; monotonicDec = $false; minV = 0.0; maxV = 0.0 }
  }
  $eps = 0.002
  $prevSign = 0
  $dirChanges = 0
  $monInc = $true
  $monDec = $true
  $minV = [double]$vals[0]
  $maxV = [double]$vals[0]
  for ($i = 1; $i -lt $vals.Length; $i++) {
    $v = [double]$vals[$i]
    if ($v -lt $minV) { $minV = $v }
    if ($v -gt $maxV) { $maxV = $v }
    $dv = $v - [double]$vals[$i - 1]
    if ($dv -gt $eps) { $monDec = $false }
    if ($dv -lt (-$eps)) { $monInc = $false }
    $s = 0
    if ($dv -gt $eps) { $s = 1 }
    elseif ($dv -lt (-$eps)) { $s = -1 }
    if ($s -ne 0) {
      if ($prevSign -ne 0 -and $s -ne $prevSign) { $dirChanges++ }
      $prevSign = $s
    }
  }
  return [pscustomobject]@{ ok = $true; dirChanges = $dirChanges; monotonicInc = $monInc; monotonicDec = $monDec; minV = $minV; maxV = $maxV }
}

function HasClearInRangeBounceNearEnd($vals, [double]$startV, [double]$endV) {
  # Detect an obvious rebound without needing overshoot outside [0,1].
  # Heuristic: after reaching close to endV, does it swing away by >= 0.02 at some later time?
  if ($null -eq $vals -or $vals.Length -lt 12) { return $false }
  $n = $vals.Length
  $reach = $false
  $reachThr = 0.0
  $awayThr = 0.0
  if ($endV -ge 0.5) {
    $reachThr = [Math]::Min(1.0, $endV - 0.01)  # reached near top
    $awayThr = [Math]::Max(0.0, $endV - 0.03)   # dipped noticeably
    for ($i = 0; $i -lt $n; $i++) {
      $v = [double]$vals[$i]
      if (-not $reach) {
        if ($v -ge $reachThr) { $reach = $true; continue }
      } else {
        if ($v -le $awayThr) { return $true }
      }
    }
  } else {
    $reachThr = [Math]::Max(0.0, $endV + 0.01)  # reached near bottom
    $awayThr = [Math]::Min(1.0, $endV + 0.03)   # rose noticeably
    for ($i = 0; $i -lt $n; $i++) {
      $v = [double]$vals[$i]
      if (-not $reach) {
        if ($v -le $reachThr) { $reach = $true; continue }
      } else {
        if ($v -ge $awayThr) { return $true }
      }
    }
  }
  return $false
}

function PromptHasBounceIntent([string]$prompt) {
  if ([string]::IsNullOrWhiteSpace($prompt)) { return $false }
  $p = $prompt.ToLowerInvariant()
  return ($p -match '(spring|bounce|bouncy|elastic|rebound|overshoot|\u5f39\u7c27|\u56de\u5f39|\u53cd\u5f39|\u5f39\u6027|\u8fc7\u51b2)')
}

function PromptHasNoBounceIntent([string]$prompt) {
  if ([string]::IsNullOrWhiteSpace($prompt)) { return $false }
  $p = $prompt.ToLowerInvariant()
  # Keep this broad; it's only used as an override when the user explicitly asks for "no bounce".
  return ($p -match '(no\\s*bounce|no\\s*overshoot|without\\s*bounce|no\\s*rebound|\u4e0d\u56de\u5f39|\u4e0d\u8981\u56de\u5f39|\u65e0\u56de\u5f39|\u4e0d\u8fc7\u51b2|\u4e0d\u8981\u8fc7\u51b2)')
}

function PromptHasPulseIntent([string]$prompt) {
  if ([string]::IsNullOrWhiteSpace($prompt)) { return $false }
  $p = $prompt.ToLowerInvariant()
  return ($p -match '(heartbeat|heart\\s*beat|pulse|\u5fc3\u8df3|\u8109\u640f|\u8109\u51b2)')
}

function TryExtractStartEndValueFromPrompt([string]$prompt) {
  $res = [ordered]@{ hasStart = $false; start = 0.0; hasEnd = $false; end = 1.0 }
  if ([string]::IsNullOrWhiteSpace($prompt)) { return [pscustomobject]$res }

  $p = $prompt

  # Build CN keywords at runtime to avoid non-ASCII literals in the script file.
  $kwStartCn = @(
    (CN @(0x5F00,0x59CB)),
    (CN @(0x8D77,0x59CB)),
    (CN @(0x5F00,0x5934)),
    (CN @(0x8D77,0x70B9))
  )
  $kwEndCn = @(
    (CN @(0x7ED3,0x675F)),
    (CN @(0x7EC8,0x70B9)),
    (CN @(0x7ED3,0x5C3E))
  )

  $fwComma = (CN @(0xFF0C))
  $fwSemi = (CN @(0xFF1B))
  $fwColon = (CN @(0xFF1A))

  $sep = "(?:^|\\s|,|;|:|$fwComma|$fwSemi|$fwColon)"
  $startKw = "(?:start|begin|from|" + (($kwStartCn | ForEach-Object { [regex]::Escape($_) }) -join "|") + ")"
  $endKw = "(?:end|to|finish|" + (($kwEndCn | ForEach-Object { [regex]::Escape($_) }) -join "|") + ")"

  # Allow a few non-digit chars between keyword and number (e.g. "end also 0.5").
  $m = [regex]::Match($p, "(?i)$sep$startKw\\s*(?:[:=]|$fwColon)?\\s*[^0-9]{0,12}(\\d*\\.?\\d+)")
  if ($m.Success) {
    $v = 0.0
    if ([double]::TryParse($m.Groups[1].Value, [ref]$v)) { $res.hasStart = $true; $res.start = (Clamp $v 0.0 1.0) }
  }

  $m2 = [regex]::Match($p, "(?i)$sep$endKw\\s*(?:[:=]|$fwColon)?\\s*[^0-9]{0,12}(\\d*\\.?\\d+)")
  if ($m2.Success) {
    $v2 = 0.0
    if ([double]::TryParse($m2.Groups[1].Value, [ref]$v2)) { $res.hasEnd = $true; $res.end = (Clamp $v2 0.0 1.0) }
  }
  return [pscustomobject]$res
}

function EnsureDslEndpoints($dsl, $specAll, [string]$prompt, $sketchTv) {
  if ($null -eq $dsl) { return $dsl }

  $sv = $null
  $ev = $null
  try { if ($null -ne $specAll -and $null -ne $specAll.spec) { $sv = $specAll.spec.start_value; $ev = $specAll.spec.end_value } } catch { }

  $pe = TryExtractStartEndValueFromPrompt $prompt
  if ($pe.hasStart) { $sv = $pe.start }
  if ($pe.hasEnd) { $ev = $pe.end }

  # If prompt didn't specify endpoints, use sketch endpoints when available.
  try {
    if (-not $pe.hasStart -and $null -eq $sv -and $null -ne $sketchTv -and $sketchTv.Count -ge 2) { $sv = [double]$sketchTv[0].v }
    if (-not $pe.hasEnd -and $null -eq $ev -and $null -ne $sketchTv -and $sketchTv.Count -ge 2) { $ev = [double]$sketchTv[$sketchTv.Count - 1].v }
  } catch { }

  # Default heartbeat/pulse baseline when the user didn't specify endpoints and no sketch is present.
  try {
    if (-not $pe.hasStart -and -not $pe.hasEnd -and ($null -eq $sketchTv -or $sketchTv.Count -lt 2) -and (PromptHasPulseIntent $prompt)) {
      $sv = 0.0
      $ev = 0.0
    }
  } catch { }

  try {
    if ($null -ne $sv) { $dsl.start_value = [double](Clamp ([double]$sv) 0.0 1.0) }
    if ($null -ne $ev) { $dsl.end_value = [double](Clamp ([double]$ev) 0.0 1.0) }
  } catch { }

  return $dsl
}

function InvokeAiCurveSpec([string]$baseUrl, [string]$model, [string]$apiKey, [string]$prompt, $sketchTv) {
  # Keep scripts ASCII-only to avoid Windows PowerShell encoding/parser issues.
  $system = @"
You are an animation curve SPEC analyzer.
Task: Based ONLY on the user's prompt, produce a curve spec for later generation. Do NOT assume you already have any curve data.
Output: Return ONLY one JSON object, with fields:
- name: string (short file-friendly name)
- analysis_cn: string (Chinese, very concise for UI; 1-3 short lines; avoid overly professional wording and avoid too many numbers; focus on style keywords + what to change)
- spec: object, must include:
  - value_min: number (0..1)
  - value_max: number (0..1)
  - start_value: number (0..1, optional; if prompt specifies start)
  - end_value: number (0..1, optional; if prompt specifies end)
  - expect_bounce: boolean (does the prompt imply bounce/overshoot/recoil?)
  - max_bounces_hint: number (0..6, can be float)
  - prefer_bounce_region: string ("front"|"end"|"any")
  - key_style: string (recommended: "extrema")
  - max_keys: integer (recommended: 6..18)
Notes:
- Do NOT include start_value/end_value unless the user explicitly specifies them (or the prompt is clearly pulse/heartbeat baseline).
- If Sketch(t,v) samples are provided, interpret what the sketch means (peaks/valleys, oscillation count, hard/soft, settle speed) and reflect it in analysis_cn/spec.
- If prompt indicates heartbeat/pulse (\u5fc3\u8df3/\u8109\u640f/\u8109\u51b2) and the user did NOT specify start/end values, set spec.start_value=0 and spec.end_value=0 (baseline), while keeping value_min=0 and value_max=1.
Return ONLY JSON. No markdown.
"@.Trim()

  $sk = ""
  try {
    if ($null -ne $sketchTv -and $sketchTv.Count -ge 8) { $sk = FormatSketchPairs $sketchTv 48 }
  } catch { $sk = "" }

  $user = "User prompt:`n$prompt`n"
  if (-not [string]::IsNullOrWhiteSpace($sk)) {
    $user = $user + "`nSketch(t,v) samples:`n$sk`n"
  }
  $user = $user + "`nReturn JSON now:"
  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user 0.18 700
  $txt = ExtractChatContentText $resp
  return (ParseJsonFromModelText $txt)
}

function InvokeAiCurveSpecRevision([string]$baseUrl, [string]$model, [string]$apiKey, [string]$basePrompt, [string]$revisionText, $prevSpecAll, $prevCurveTv, $sketchTv) {
  $system = @"
You are an animation curve SPEC reviser.
Task: Update the previous CurveSpec based ONLY on: (1) base prompt, (2) user's revision request, (3) previous CurveSpec, and optionally (4) previous curve samples/sketch.
Output: Return ONLY one JSON object with fields:
- name: string
- analysis_cn: string (short Chinese summary of the NEW intent, 1-3 lines)
- spec: object, must include:
  - value_min (0..1), value_max (0..1)
  - start_value/end_value (0..1, optional; only if requested)
  - expect_bounce (boolean), max_bounces_hint (0..6), prefer_bounce_region ("front"|"end"|"any")
  - key_style (recommended: "extrema"), max_keys (recommended: 6..18)
Notes:
- If Sketch(t,v) samples are provided, treat them as semantic reference (what it means), not necessarily a hard constraint, and reflect it in the updated intent.
- If intent indicates heartbeat/pulse and user did NOT specify endpoints, keep start_value=end_value=0 while value_min=0,value_max=1.
Return ONLY JSON. No markdown.
"@.Trim()

  $prevJson = "{}"
  try { if ($null -ne $prevSpecAll) { $prevJson = ($prevSpecAll | ConvertTo-Json -Depth 10) } } catch { $prevJson = "{}" }

  $prevCurve = ""
  try { if ($null -ne $prevCurveTv -and $prevCurveTv.Count -ge 8) { $prevCurve = FormatSketchPairs $prevCurveTv 48 } } catch { $prevCurve = "" }
  $sk = ""
  try { if ($null -ne $sketchTv -and $sketchTv.Count -ge 8) { $sk = FormatSketchPairs $sketchTv 48 } } catch { $sk = "" }

  $user = @"
BasePrompt:
$basePrompt

UserRevision:
$revisionText

PreviousSpec(JSON):
$prevJson
"@.Trim()

  if (-not [string]::IsNullOrWhiteSpace($prevCurve)) {
    $user = $user + "`n`nPreviousCurve(t,v) samples:`n" + $prevCurve
  }
  if (-not [string]::IsNullOrWhiteSpace($sk)) {
    $user = $user + "`n`nSketch(t,v) samples:`n" + $sk
  }
  $user = $user + "`n`nReturn JSON now:"

  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user 0.18 700
  $txt = ExtractChatContentText $resp
  return (ParseJsonFromModelText $txt)
}

function InvokeAiDslFromPromptAndSpec([string]$baseUrl, [string]$model, [string]$apiKey, [string]$prompt, $specObj, $sketchTv, [string]$fixHint) {
  $system = @"
You are an animation curve DSL generator. Produce a DSL (JSON) that will be converted to a .rascurve.
Output requirements (VERY IMPORTANT):
- Return ONLY one JSON object. No extra text, no markdown.
- Must be: version=2, model="stack"
- MUST include a "base" object (stack DSL requires "base"), and base.model must be set.
- If start_value == end_value (e.g. 0.5 -> 0.5) and bounce is expected, generate a curve that oscillates around that center value (use spring around center with amplitude_value, or use a ring layer on top of a hold base).
- Values must stay within [0,1] (you may use range_mode="soft", but final curve must still be in 0..1)
 - key_style is recommended to be "extrema" and max_keys should be in [6..18]
 - If the prompt/spec expects bounce/overshoot, try to include at least one visible rebound (peak/valley), but do not force extra oscillations.
- IMPORTANT: If end_value is near a hard bound (0 or 1), do NOT rely on overshooting beyond the bound (it will get clamped away). Prefer an in-range "undershoot" bounce:
  - For end_value ~ 1: approach 1 quickly, then dip slightly below 1, then settle back to 1 (all within 0..1).
  - For end_value ~ 0: approach 0 quickly, then rise slightly above 0, then settle back to 0 (all within 0..1).
- IMPORTANT: If the user prompt itself contains bounce intent words (spring/bounce/elastic/overshoot or Chinese equivalents), treat it as bounce intent even if CurveSpec.expect_bounce is false.
- IMPORTANT: If the prompt indicates heartbeat/pulse (\u5fc3\u8df3/\u8109\u640f/\u8109\u51b2), generate a pulse-like curve: narrow main spike + smaller secondary bump shortly after, baseline near 0 between beats. Prefer impact/ring layers with amplitude_value (absolute) so the peak reaches close to 1 when the target range is 0..1.
You may use:
- base model: ease|attract|spring|linear|hold
- optional layers: impact|ring|noise|oscillate
- optional envelope/follow
Return ONLY DSL JSON.
"@.Trim()

  $specJson = ""
  try { $specJson = ($specObj | ConvertTo-Json -Depth 8) } catch { $specJson = "{}" }

  $analysisCn = ""
  try { $analysisCn = [string]$specObj.analysis_cn } catch { $analysisCn = "" }

  $sk2 = ""
  try {
    if ($null -ne $sketchTv -and $sketchTv.Count -ge 8) { $sk2 = FormatSketchPairs $sketchTv 64 }
  } catch { $sk2 = "" }

  $user = @"
Prompt:
$prompt

CurveSpec(JSON):
$specJson
"@.Trim()

  if (-not [string]::IsNullOrWhiteSpace($analysisCn)) {
    $user = $user + "`n`nAnalysisCN (authoritative intent description):`n" + $analysisCn.Trim()
  }

  if (-not [string]::IsNullOrWhiteSpace($sk2)) {
    $user = $user + "`n`nSketch(t,v) samples (must match shape):`n" + $sk2
  }

  if (-not [string]::IsNullOrWhiteSpace($fixHint)) {
    $user = $user + "`n`nFix requirements:`n" + $fixHint.Trim()
  }

  $user = $user + "`n`nReturn DSL JSON now:"
  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user 0.22 1000
  $txt = ExtractChatContentText $resp
  $dsl = ParseJsonFromModelText $txt
  return $dsl
}

function InvokeAiKeysFromPromptAndSpec([string]$baseUrl, [string]$model, [string]$apiKey, [string]$prompt, $specObj, $sketchTv, [string]$fixHint) {
  # "Free mode": generate (t,v) keyframes directly, then we serialize a valid .rascurve locally.
  $system = @"
You are an animation curve KEYFRAME generator.
Task: Based on the user's prompt + CurveSpec, output key points that match the intended curve.
Output requirements (VERY IMPORTANT):
- Return ONLY one JSON object. No extra text, no markdown.
- Must include: keys: array of objects {t:number, v:number}
- t is normalized time in [0,1], v is normalized value in [0,1]
- Key count: 6..32 (more keys around sharp peaks, fewer elsewhere)
- Keys should focus on important turning points (peaks/valleys) and timing beats, but you may add extra points if needed for shape.
- If CurveSpec.spec.start_value is provided, first key MUST be t=0 with v=start_value.
- If CurveSpec.spec.end_value is provided, last key MUST be t=1 with v=end_value.
- If the prompt indicates heartbeat/pulse, produce a pulse-like shape: narrow main spike + smaller secondary bump shortly after, baseline near 0 between beats.
- If the user is revising an existing curve and PreviousCurve samples are provided, use it as a reference; prioritize the user's new request.
Return ONLY JSON.
"@.Trim()

  $specJson = ""
  try { $specJson = ($specObj | ConvertTo-Json -Depth 8) } catch { $specJson = "{}" }

  $analysisCn = ""
  try { $analysisCn = [string]$specObj.analysis_cn } catch { $analysisCn = "" }

  $sk2 = ""
  try {
    if ($null -ne $sketchTv -and $sketchTv.Count -ge 8) { $sk2 = FormatSketchPairs $sketchTv 64 }
  } catch { $sk2 = "" }

  $user = @"
Prompt:
$prompt

CurveSpec(JSON):
$specJson
"@.Trim()

  if (-not [string]::IsNullOrWhiteSpace($analysisCn)) {
    $user = $user + "`n`nAnalysisCN (authoritative intent description):`n" + $analysisCn.Trim()
  }

  # When revising, the caller may embed PreviousCurve samples into the prompt.

  if (-not [string]::IsNullOrWhiteSpace($sk2)) {
    $user = $user + "`n`nSketch(t,v) samples (must match shape):`n" + $sk2
  }

  if (-not [string]::IsNullOrWhiteSpace($fixHint)) {
    $user = $user + "`n`nFix requirements:`n" + $fixHint.Trim()
  }

  $user = $user + "`n`nReturn JSON now:"
  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user 0.22 1000
  $txt = ExtractChatContentText $resp
  return (ParseJsonFromModelText $txt)
}

function InvokeAiCurveOneShot([string]$baseUrl, [string]$model, [string]$apiKey, [string]$intentText, $sketchTv, [string]$sketchMode, $prevCurveTv, [string]$fixHint, [string]$chatText) {
  # One-shot "chat-like" generation: analysis + keys come from the same model response (keeps them consistent).
  $system = @"
 You are an animation curve designer.
 Task: Generate an animation curve that matches the user's intent, like a helpful chat assistant would.
 Treat ChatContext as multi-turn dialogue memory and prioritize the latest user request while preserving relevant style continuity.
 Output: Return ONLY one JSON object with fields:
 - name: string (short file-friendly)
 - analysis_cn: string (Chinese, very short; 1-3 lines; simple words)
 - keys: array of objects {t:number, v:number, s:number?, inWeight:number?, outWeight:number?}
 Rules:
 - t is normalized time [0,1], v is normalized value [0,1]. (Do NOT output v outside 0..1.)
 - Use as many keys as needed to match the feel (usually 4-64). If the curve has sharp spikes or quick rebounds, add a few support keys around the spike so it looks smooth (not polygonal).
 - Prefer natural, chat-like interpretation of intent. Do not over-compress into rigid templates unless the user explicitly asks for strict structure.
 - IMPORTANT (revision/local edit mode): if intent contains UserRevision and PreviousCurve samples are provided, treat it as local editing. Keep overall timing rhythm / major peak layout close to PreviousCurve, and only apply targeted adjustments requested by user.
 - If the intent implies spring/bounce, you MAY include rebounds, but it is NOT mandatory (a very stiff spring can look like a fast settle).
 - If the intent is heartbeat/pulse: narrow main spike + smaller second bump shortly after; baseline near 0 between beats.
 - If SketchMode is ""reference"": treat sketch as meaning only (style/peaks count), do NOT try to fit exact points.
 - If SketchMode is ""constraint"": match the sketch shape closely (within 0..1).
 - You MAY provide an optional slope s per key. You MAY also provide optional inWeight/outWeight per key (typical 0.15..0.6). Smaller weights make corners tighter; bigger weights look smoother.
 Return ONLY JSON. No markdown.
"@.Trim()

  $sk = ""
  try { if ($null -ne $sketchTv -and $sketchTv.Count -ge 8) { $sk = FormatSketchPairs $sketchTv 64 } } catch { $sk = "" }
  $pc = ""
  try { if ($null -ne $prevCurveTv -and $prevCurveTv.Count -ge 8) { $pc = FormatSketchPairs $prevCurveTv 64 } } catch { $pc = "" }

  if ([string]::IsNullOrWhiteSpace($sketchMode)) { $sketchMode = "none" }

  $user = @"
Intent:
$intentText

SketchMode:
$sketchMode
"@.Trim()

  if (-not [string]::IsNullOrWhiteSpace($chatText)) { $user = $user + "`n`nChatContext:`n" + $chatText }
  if (-not [string]::IsNullOrWhiteSpace($sk)) { $user = $user + "`n`nSketch(t,v) samples:`n" + $sk }
  if (-not [string]::IsNullOrWhiteSpace($pc)) { $user = $user + "`n`nPreviousCurve(t,v) samples:`n" + $pc }
  if (-not [string]::IsNullOrWhiteSpace($fixHint)) { $user = $user + "`n`nFix requirements:`n" + $fixHint.Trim() }

  $user = $user + "`n`nReturn JSON now:"

  $temp = 0.68
  if ($intentText -match "(?is)UserRevision\\s*:") { $temp = 0.36 }
  if ($temp -gt 0.50 -and $null -ne $prevCurveTv -and $prevCurveTv.Count -ge 8) { $temp = 0.50 }
  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user $temp 1800
  $txt = ExtractChatContentText $resp
  return (ParseJsonFromModelText $txt)
}

function InvokeAiCurveVariantsOneShot([string]$baseUrl, [string]$model, [string]$apiKey, [string]$intentText, [int]$variantsCount, [string]$variantStylesText, $sketchTv, [string]$sketchMode, $prevCurveTv, $selectedCurveTv, [string]$fixHint, [string]$chatText) {
  $variantsCount = [Math]::Max(1, [Math]::Min(5, $variantsCount))
  $system = @"
You are an animation curve designer.
Task: Generate MULTIPLE curve options ("draw cards") that match the user's intent, like a helpful chat assistant would.
Treat ChatContext as conversation memory; keep intent continuity but allow meaningful creative variation across cards.
 Output: Return ONLY one JSON object with fields:
 - variants: array of length N, each item must include:
   - name: string
   - analysis_cn: string (Chinese, very short; 1-2 lines; simple words)
   - keys: array of {t:number, v:number, s:number?, inWeight:number?, outWeight:number?}
 Rules:
- N must equal VariantsCount.
- t in [0,1], v in [0,1]. Do NOT output v outside 0..1.
- Each variant should feel meaningfully different (timing / stiffness / bounces count), not just tiny tweaks.
 - Keep variant personality natural and conversational; avoid forcing the same formula across cards.
- If intent implies spring/bounce, variants MAY include rebounds, but it is NOT mandatory (a very stiff spring can be a fast settle).
 - If intent is heartbeat/pulse: narrow main spike + smaller second bump shortly after; baseline near 0 between beats.
 - If SketchMode is ""reference"": treat sketch as meaning only; do NOT fit exact points.
 - If SketchMode is ""constraint"": match the sketch closely (within 0..1).
 Return ONLY JSON. No markdown.
"@.Trim()

  if ([string]::IsNullOrWhiteSpace($sketchMode)) { $sketchMode = "none" }

  $sk = ""
  try { if ($null -ne $sketchTv -and $sketchTv.Count -ge 8) { $sk = FormatSketchPairs $sketchTv 64 } } catch { $sk = "" }
  $pc = ""
  try { if ($null -ne $prevCurveTv -and $prevCurveTv.Count -ge 8) { $pc = FormatSketchPairs $prevCurveTv 64 } } catch { $pc = "" }
  $sc = ""
  try { if ($null -ne $selectedCurveTv -and $selectedCurveTv.Count -ge 8) { $sc = FormatSketchPairs $selectedCurveTv 64 } } catch { $sc = "" }

  $user = @"
Intent:
$intentText

VariantsCount:
$variantsCount

VariantStyles (optional guidance):
$variantStylesText

SketchMode:
$sketchMode
"@.Trim()

  if (-not [string]::IsNullOrWhiteSpace($chatText)) { $user = $user + "`n`nChatContext:`n" + $chatText }
  if (-not [string]::IsNullOrWhiteSpace($sk)) { $user = $user + "`n`nSketch(t,v) samples:`n" + $sk }
  if (-not [string]::IsNullOrWhiteSpace($pc)) { $user = $user + "`n`nPreviousCurve(t,v) samples:`n" + $pc }
  if (-not [string]::IsNullOrWhiteSpace($sc)) { $user = $user + "`n`nCurrentSelectedCurve(t,v) samples (optional reference):`n" + $sc }
  if (-not [string]::IsNullOrWhiteSpace($fixHint)) { $user = $user + "`n`nFix requirements:`n" + $fixHint.Trim() }
  $user = $user + "`n`nReturn JSON now:"

  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user 0.78 2400
  $txt = ExtractChatContentText $resp
  $obj = ParseJsonFromModelText $txt $false
  # Some models return the variants array as the top-level JSON. Accept it for robustness.
  if ($obj -is [System.Array]) { return [pscustomobject]@{ variants = $obj } }
  return $obj
}

function FormatChatTurnsForAi($turns, [int]$maxChars) {
  $maxChars = [Math]::Max(200, [Math]::Min(8000, $maxChars))
  if ($null -eq $turns -or -not ($turns -is [System.Array]) -or $turns.Count -le 0) { return "" }
  $sb = New-Object System.Text.StringBuilder
  foreach ($t in $turns) {
    $role = ""
    $text = ""
    try { $role = [string]$t.role } catch { $role = "" }
    try { $text = [string]$t.text } catch { $text = "" }
    if ([string]::IsNullOrWhiteSpace($text)) { continue }
    $role = $role.Trim().ToLowerInvariant()
    if ($role -ne "user" -and $role -ne "assistant") { $role = "user" }
    [void]$sb.Append($role)
    [void]$sb.Append(": ")
    $line = $text.Trim()
    if ($line.Length -gt 600) { $line = $line.Substring(0,600) + "..." }
    [void]$sb.AppendLine($line)
    [void]$sb.AppendLine("")
  }
  $s = $sb.ToString().Trim()
  if ($s.Length -gt $maxChars) { $s = $s.Substring($s.Length - $maxChars) }
  return $s
}

function InvokeAiCurveCritique([string]$baseUrl, [string]$model, [string]$apiKey, [string]$intentText, $specAll, $curveTv, $sketchTv, [string]$sketchMode, [string]$fixHint, [string]$chatText) {
  # Internal agent step: decide if the curve matches intent; if not, propose a short fix hint for the next attempt.
  $system = @"
You are a curve-accuracy critic.
Task: Check whether the generated curve matches the user's intent (intent text + CurveSpec + optional sketch meaning).
Output: Return ONLY one JSON object with fields:
- ok: boolean
- fix_hint: string (very short, 1-2 lines, actionable; empty if ok)
Rules:
- Do NOT mention policy or formatting.
- Prefer minimal changes: change timing, peak count, stiffness, settle, amplitude.
- If SketchMode is "reference", treat sketch as meaning only; do NOT force matching its exact points.
- If SketchMode is "constraint", require closer matching to sketch shape.
Return ONLY JSON.
"@.Trim()

  $specJson = "{}"
  try { $specJson = ($specAll | ConvertTo-Json -Depth 10) } catch { $specJson = "{}" }
  $c = ""
  try { if ($null -ne $curveTv -and $curveTv.Count -ge 8) { $c = FormatSketchPairs $curveTv 64 } } catch { $c = "" }
  $sk = ""
  try { if ($null -ne $sketchTv -and $sketchTv.Count -ge 8) { $sk = FormatSketchPairs $sketchTv 48 } } catch { $sk = "" }
  if ([string]::IsNullOrWhiteSpace($sketchMode)) { $sketchMode = "none" }

  $user = @"
Intent:
$intentText

CurveSpec(JSON):
$specJson

SketchMode:
$sketchMode
"@.Trim()

  if (-not [string]::IsNullOrWhiteSpace($chatText)) { $user = $user + "`n`nChatContext:`n" + $chatText }
  if (-not [string]::IsNullOrWhiteSpace($sk)) { $user = $user + "`n`nSketch(t,v) samples:`n" + $sk }
  if (-not [string]::IsNullOrWhiteSpace($c)) { $user = $user + "`n`nGeneratedCurve(t,v) samples:`n" + $c }
  if (-not [string]::IsNullOrWhiteSpace($fixHint)) { $user = $user + "`n`nCurrentFixHint:`n" + $fixHint.Trim() }
  $user = $user + "`n`nReturn JSON now:"

  $resp = InvokeChatCompletions $baseUrl $model $apiKey $system $user 0.2 450
  $txt = ExtractChatContentText $resp
  return (ParseJsonFromModelText $txt)
}

function ExtractAiTvKeys($obj) {
  if ($null -eq $obj) { return @() }
  $arr = $null
  try { $arr = $obj.keys } catch { $arr = $null }
  if ($null -eq $arr) { try { $arr = $obj.points } catch { $arr = $null } }
  if ($null -eq $arr) { try { $arr = $obj.anchors } catch { $arr = $null } }
  if ($null -eq $arr) { try { $arr = $obj.tv } catch { $arr = $null } }
  if ($null -eq $arr -or -not ($arr -is [System.Array])) { return @() }
  return $arr
}

function NormalizeAndSlopeKeys($tvArr) {
  # Returns key objects: {t,v,s,inWeight,outWeight} with stable ordering, dedup, edge padding, and slope estimation.
  if ($null -eq $tvArr -or $tvArr.Count -lt 2) { return @() }

  $tmp = New-Object System.Collections.Generic.List[object]
  foreach ($p in $tvArr) {
    $t = $null; $v = $null; $sIn = $null
    $wIn = $null; $wOut = $null; $wBoth = $null
    try { $t = [double]$p.t } catch { try { $t = [double]$p[0] } catch { $t = $null } }
    try { $v = [double]$p.v } catch { try { $v = [double]$p[1] } catch { $v = $null } }
    try { $sIn = [double]$p.s } catch { $sIn = $null }
    try {
      [void](TryGetDoubleProp $p @("inWeight","inW","wIn","w_in","win") ([ref]$wIn))
      [void](TryGetDoubleProp $p @("outWeight","outW","wOut","w_out","wout") ([ref]$wOut))
      [void](TryGetDoubleProp $p @("w","weight","handleWeight") ([ref]$wBoth))
      if ($null -ne $wBoth) {
        if ($null -eq $wIn) { $wIn = $wBoth }
        if ($null -eq $wOut) { $wOut = $wBoth }
      }
    } catch { }
    if ($null -eq $t -or $null -eq $v) { continue }
    if ([double]::IsNaN($t) -or [double]::IsInfinity($t) -or [double]::IsNaN($v) -or [double]::IsInfinity($v)) { continue }
    $obj = [pscustomobject]@{ t = (Clamp $t 0.0 1.0); v = (Clamp $v 0.0 1.0) }
    if ($null -ne $sIn -and -not [double]::IsNaN($sIn) -and -not [double]::IsInfinity($sIn)) {
      try { $obj | Add-Member -NotePropertyName "s_in" -NotePropertyValue ([double]$sIn) -Force } catch { }
    }
    if ($null -ne $wIn -and -not [double]::IsNaN($wIn) -and -not [double]::IsInfinity($wIn)) {
      try { $obj | Add-Member -NotePropertyName "w_in" -NotePropertyValue ([double]$wIn) -Force } catch { }
    }
    if ($null -ne $wOut -and -not [double]::IsNaN($wOut) -and -not [double]::IsInfinity($wOut)) {
      try { $obj | Add-Member -NotePropertyName "w_out" -NotePropertyValue ([double]$wOut) -Force } catch { }
    }
    $tmp.Add($obj) | Out-Null
  }
  if ($tmp.Count -lt 2) { return @() }

  $sorted = @($tmp | Sort-Object t)
  $ded = New-Object System.Collections.Generic.List[object]
  foreach ($k in $sorted) {
    if ($ded.Count -eq 0) { $ded.Add($k) | Out-Null; continue }
    $last = $ded[$ded.Count - 1]
    if ([Math]::Abs([double]$k.t - [double]$last.t) -lt 1e-5) { $ded[$ded.Count - 1] = $k; continue }
    $ded.Add($k) | Out-Null
  }
  if ($ded.Count -lt 2) { return @() }

  # Pad edges so preview domain is complete.
  if ([double]$ded[0].t -gt 1e-6) { $ded.Insert(0, [pscustomobject]@{ t = 0.0; v = [double]$ded[0].v }) }
  if ([double]$ded[$ded.Count - 1].t -lt (1.0 - 1e-6)) { $ded.Add([pscustomobject]@{ t = 1.0; v = [double]$ded[$ded.Count - 1].v }) | Out-Null }

  # Cap count defensively (AI can sometimes dump too many points).
  if ($ded.Count -gt 40) { $ded = @($ded | Select-Object -First 40) }

  $n = $ded.Count

  # Slope estimation:
  # - Use monotone cubic tangents (Fritsch-Carlson) as a safe default (prevents overshoot -> avoids clamping artifacts).
  # - If the model provides per-key slope s, prefer it (still clamp + keep control points in range).
  $tArr = New-Object double[] $n
  $vArr = New-Object double[] $n
  $hasAi = New-Object bool[] $n
  $aiS = New-Object double[] $n
  $inW = New-Object double[] $n
  $outW = New-Object double[] $n
  for ($i = 0; $i -lt $n; $i++) {
    $tArr[$i] = [double]$ded[$i].t
    $vArr[$i] = [double]$ded[$i].v
    $hasAi[$i] = $false
    $aiS[$i] = 0.0
    $inW[$i] = 1.0 / 3.0
    $outW[$i] = 1.0 / 3.0
    try {
      $pin = $ded[$i].PSObject.Properties["s_in"]
      if ($pin -and $null -ne $pin.Value) {
        $aiS[$i] = [double]$pin.Value
        $hasAi[$i] = $true
      }
    } catch { }
    try {
      $pwi = $ded[$i].PSObject.Properties["w_in"]
      if ($pwi -and $null -ne $pwi.Value) { $inW[$i] = Clamp ([double]$pwi.Value) 0.0 1.0 }
    } catch { }
    try {
      $pwo = $ded[$i].PSObject.Properties["w_out"]
      if ($pwo -and $null -ne $pwo.Value) { $outW[$i] = Clamp ([double]$pwo.Value) 0.0 1.0 }
    } catch { }
  }

  # Prevent invalid handle overlap: ensure outW[i] + inW[i+1] < 1 for each segment.
  for ($i = 0; $i -lt ($n - 1); $i++) {
    $sum = $outW[$i] + $inW[$i + 1]
    if ($sum -gt 0.999) {
      $scale = 0.999 / $sum
      $outW[$i] = Clamp ($outW[$i] * $scale) 0.0 1.0
      $inW[$i + 1] = Clamp ($inW[$i + 1] * $scale) 0.0 1.0
    }
  }

  $m = New-Object double[] $n
  for ($i = 0; $i -lt $n; $i++) { $m[$i] = 0.0 }
  if ($n -ge 2) {
    $d = New-Object double[] ($n - 1)
    for ($i = 0; $i -lt ($n - 1); $i++) {
      $dt = $tArr[$i + 1] - $tArr[$i]
      if ($dt -gt 1e-9) { $d[$i] = ($vArr[$i + 1] - $vArr[$i]) / $dt } else { $d[$i] = 0.0 }
    }

    $m[0] = $d[0]
    $m[$n - 1] = $d[$n - 2]
    for ($i = 1; $i -le ($n - 2); $i++) {
      $m[$i] = 0.5 * ($d[$i - 1] + $d[$i])
      if (($d[$i - 1] * $d[$i]) -le 0.0) { $m[$i] = 0.0 }
    }

    # Fritsch-Carlson limiter (monotone on each interval when d keeps sign).
    for ($i = 0; $i -lt ($n - 1); $i++) {
      $di = $d[$i]
      if ([Math]::Abs($di) -lt 1e-12) {
        $m[$i] = 0.0
        $m[$i + 1] = 0.0
        continue
      }
      $a = $m[$i] / $di
      $b = $m[$i + 1] / $di
      if ($a -lt 0.0 -or $b -lt 0.0) {
        $m[$i] = 0.0
        $m[$i + 1] = 0.0
        continue
      }
      $h = ($a * $a) + ($b * $b)
      if ($h -gt 9.0) {
        $tau = 3.0 / [Math]::Sqrt($h)
        $m[$i] = $tau * $a * $di
        $m[$i + 1] = $tau * $b * $di
      }
    }
  }

  # Apply AI-provided slopes (if any).
  for ($i = 0; $i -lt $n; $i++) {
    if ($hasAi[$i]) { $m[$i] = $aiS[$i] }
  }

  # Keep slopes reasonable; prevents huge overshoot from bezier tangents.
  $sLim = 20.0
  for ($i = 0; $i -lt $n; $i++) {
    $s = $m[$i]
    if ([double]::IsNaN($s) -or [double]::IsInfinity($s)) { $s = 0.0 }
    $m[$i] = Clamp $s (-$sLim) $sLim
  }

  # Keep bezier control point y1/y2 in [0,1] (reduces "kink" artifacts caused by clamping samples).
  # Handle dx uses per-key weights.
  for ($pass = 0; $pass -lt 2; $pass++) {
    for ($i = 0; $i -lt ($n - 1); $i++) {
      $dtSeg = $tArr[$i + 1] - $tArr[$i]
      if ($dtSeg -le 1e-9) { continue }
      $y0 = $vArr[$i]
      $y3 = $vArr[$i + 1]

      $dx0 = $dtSeg * (Clamp $outW[$i] 0.0 1.0)
      if ($dx0 -gt 1e-9) {
        $y1 = $y0 + $m[$i] * $dx0
        if ($y1 -lt 0.0) { $m[$i] = (0.0 - $y0) / $dx0 }
        elseif ($y1 -gt 1.0) { $m[$i] = (1.0 - $y0) / $dx0 }
      }

      $dx1 = $dtSeg * (Clamp $inW[$i + 1] 0.0 1.0)
      if ($dx1 -gt 1e-9) {
        $y2 = $y3 - $m[$i + 1] * $dx1
        if ($y2 -lt 0.0) { $m[$i + 1] = ($y3 - 0.0) / $dx1 }
        elseif ($y2 -gt 1.0) { $m[$i + 1] = ($y3 - 1.0) / $dx1 }
      }
    }
  }
  for ($i = 0; $i -lt $n; $i++) { $m[$i] = Clamp $m[$i] (-$sLim) $sLim }

  $out = New-Object System.Collections.Generic.List[object]
  for ($i = 0; $i -lt $n; $i++) {
    $out.Add([pscustomobject]@{
      t = [double]$tArr[$i]
      v = [double]$vArr[$i]
      s = [double]$m[$i]
      inWeight = [double](Clamp $inW[$i] 0.0 1.0)
      outWeight = [double](Clamp $outW[$i] 0.0 1.0)
    }) | Out-Null
  }
  return $out.ToArray()
}

function ResampleTvLinear($tvArr, [int]$count) {
  $count = [Math]::Max(8, [Math]::Min(256, $count))
  if ($null -eq $tvArr) { return @() }

  $tmp = New-Object System.Collections.Generic.List[object]
  foreach ($p in $tvArr) {
    try {
      $t = [double]$p.t
      $v = [double]$p.v
      if ([double]::IsNaN($t) -or [double]::IsInfinity($t) -or [double]::IsNaN($v) -or [double]::IsInfinity($v)) { continue }
      $tmp.Add([pscustomobject]@{ t = (Clamp $t 0.0 1.0); v = (Clamp $v 0.0 1.0) }) | Out-Null
    } catch { }
  }
  if ($tmp.Count -lt 2) { return @() }

  $sorted = @($tmp | Sort-Object t)
  $ded = New-Object System.Collections.Generic.List[object]
  foreach ($k in $sorted) {
    if ($ded.Count -eq 0) { $ded.Add($k) | Out-Null; continue }
    $last = $ded[$ded.Count - 1]
    if ([Math]::Abs([double]$k.t - [double]$last.t) -lt 1e-6) {
      $ded[$ded.Count - 1] = $k
      continue
    }
    $ded.Add($k) | Out-Null
  }
  if ($ded.Count -lt 2) { return @() }

  if ([double]$ded[0].t -gt 1e-9) { $ded.Insert(0, [pscustomobject]@{ t = 0.0; v = [double]$ded[0].v }) }
  if ([double]$ded[$ded.Count - 1].t -lt (1.0 - 1e-9)) { $ded.Add([pscustomobject]@{ t = 1.0; v = [double]$ded[$ded.Count - 1].v }) | Out-Null }

  $out = New-Object System.Collections.Generic.List[object]
  $seg = 0
  for ($i = 0; $i -lt $count; $i++) {
    $t = if ($count -le 1) { 0.0 } else { [double]$i / [double]($count - 1) }
    while ($seg + 1 -lt $ded.Count -and [double]$ded[$seg + 1].t -lt $t) { $seg++ }
    if ($seg + 1 -ge $ded.Count) { $seg = $ded.Count - 2 }
    $a = $ded[$seg]
    $b = $ded[$seg + 1]
    $ta = [double]$a.t
    $tb = [double]$b.t
    $va = [double]$a.v
    $vb = [double]$b.v
    $u = 0.0
    $dt = $tb - $ta
    if ($dt -gt 1e-9) { $u = (Clamp (($t - $ta) / $dt) 0.0 1.0) }
    $v = $va + ($vb - $va) * $u
    $out.Add([pscustomobject]@{ t = $t; v = (Clamp $v 0.0 1.0) }) | Out-Null
  }
  return $out.ToArray()
}

function RevisionRequestsLargeReshape([string]$revisionText) {
  if ([string]::IsNullOrWhiteSpace($revisionText)) { return $false }
  $s = $revisionText.Trim().ToLowerInvariant()
  if ($s -match '(?i)from scratch|rebuild|redo|completely|totally|different|new style|new shape|start over') { return $true }
  if ($s -match '\u91CD\u505A|\u91CD\u6765|\u5B8C\u5168|\u5F7B\u5E95|\u5927\u6539|\u6362\u4E00\u79CD|\u6362\u98CE\u683C|\u6362\u4E2A|\u6539\u6210\u53E6\u4E00\u4E2A') { return $true }
  return $false
}

function ConstrainRevisionKeysToPrevious($newKeys, $prevCurveTv, [double]$newWeight, [double]$maxDelta) {
  if ($null -eq $newKeys -or $newKeys.Count -lt 2) { return $newKeys }
  if ($null -eq $prevCurveTv -or $prevCurveTv.Count -lt 4) { return $newKeys }

  $cand = ResampleTvLinear $newKeys 96
  $prev = ResampleTvLinear $prevCurveTv 96
  if ($null -eq $cand -or $cand.Count -lt 8 -or $null -eq $prev -or $prev.Count -ne $cand.Count) { return $newKeys }

  $newWeight = Clamp $newWeight 0.10 0.95
  $maxDelta = Clamp $maxDelta 0.08 0.60

  $mae = 0.0
  for ($i = 0; $i -lt $cand.Count; $i++) {
    $mae += [Math]::Abs([double]$cand[$i].v - [double]$prev[$i].v)
  }
  $mae /= [double]$cand.Count
  if ($mae -lt 0.015) { return $newKeys }

  $out = New-Object System.Collections.Generic.List[object]
  for ($i = 0; $i -lt $cand.Count; $i++) {
    $t = [double]$cand[$i].t
    $pv = [double]$prev[$i].v
    $cv = [double]$cand[$i].v
    $v = $pv + $newWeight * ($cv - $pv)
    $dv = $v - $pv
    if ([Math]::Abs($dv) -gt $maxDelta) {
      $v = $pv + ([Math]::Sign($dv) * $maxDelta)
    }
    $out.Add([pscustomobject]@{ t = $t; v = (Clamp $v 0.0 1.0) }) | Out-Null
  }

  $norm = NormalizeAndSlopeKeys $out.ToArray()
  if ($null -eq $norm -or $norm.Count -lt 2) { return $newKeys }
  return $norm
}

function MakeLocalEaseKeys([double]$startV, [double]$endV, [string]$flavor) {
  # Simple smoothstep-based ease. Flavor is a hint only; keep it robust for local-only fallback.
  function SmoothStep([double]$x) {
    $u = Clamp $x 0.0 1.0
    return $u * $u * (3.0 - 2.0 * $u)
  }

  $tv = New-Object System.Collections.Generic.List[object]
  foreach ($t in @(0.0, 0.15, 0.35, 0.65, 0.85, 1.0)) {
    $u = $t
    if ($flavor -match 'fast' -or $flavor -match 'quick' -or $flavor -match '\u5feb') {
      # Bias to reach target earlier.
      $u = Clamp ([Math]::Pow($u, 0.65)) 0.0 1.0
    } elseif ($flavor -match 'slow' -or $flavor -match '\u6162') {
      $u = Clamp ([Math]::Pow($u, 1.25)) 0.0 1.0
    }
    $e = SmoothStep $u
    $v = $startV + ($endV - $startV) * $e
    $tv.Add([pscustomobject]@{ t = $t; v = (Clamp $v 0.0 1.0) }) | Out-Null
  }
  return (NormalizeAndSlopeKeys $tv.ToArray())
}

function MakeLocalInRangeSpringKeys([double]$startV, [double]$endV, [string]$flavor) {
  # A "hard spring" feel without leaving [0,1]: hit target quickly, dip/overshoot slightly within range, then settle.
  $d = ($endV - $startV)
  if ([Math]::Abs($d) -lt 1e-6) {
    return @([pscustomobject]@{ t = 0.0; v = (Clamp $startV 0.0 1.0); s = 0.0 }, [pscustomobject]@{ t = 1.0; v = (Clamp $endV 0.0 1.0); s = 0.0 })
  }
  $sgn = if ($d -ge 0) { 1.0 } else { -1.0 }

  $hard = ($flavor -match 'hard' -or $flavor -match 'stiff' -or $flavor -match '\u786c')
  $peakT = if ($hard) { 0.08 } else { 0.14 }
  $reboundT = if ($hard) { 0.16 } else { 0.22 }
  $settleT = if ($hard) { 0.28 } else { 0.36 }
  $amp = if ($hard) { 0.10 } else { 0.06 }

  $tv = @(
    [pscustomobject]@{ t = 0.0; v = $startV }
    [pscustomobject]@{ t = $peakT; v = $endV }
    [pscustomobject]@{ t = $reboundT; v = (Clamp ($endV - $sgn * $amp) 0.0 1.0) }
    [pscustomobject]@{ t = $settleT; v = $endV }
    [pscustomobject]@{ t = 1.0; v = $endV }
  )
  return (NormalizeAndSlopeKeys $tv)
}

function WriteTempDslAndGenerateRas([string]$dslToRasPath, $dslObj) {
  $tmpDir = Join-Path -Path $PSScriptRoot -ChildPath "out"
  if (-not (Test-Path -LiteralPath $tmpDir)) { New-Item -ItemType Directory -Force -Path $tmpDir | Out-Null }
  $dslPath = Join-Path -Path $tmpDir -ChildPath ("_ai_dsl_" + [Guid]::NewGuid().ToString("N") + ".json")
  $rasPath = Join-Path -Path $tmpDir -ChildPath ("_preview_" + [Guid]::NewGuid().ToString("N") + ".rascurve")
  $utf8NoBom = New-Object System.Text.UTF8Encoding($false)
  [System.IO.File]::WriteAllText($dslPath, (($dslObj | ConvertTo-Json -Depth 12).Trim() + "`n"), $utf8NoBom)
  & $dslToRasPath -In $dslPath -Out $rasPath -Format curve | Out-Null
  if (-not (Test-Path -LiteralPath $rasPath)) { throw "Failed to generate rascurve from DSL." }
  $rasText = [string](ReadUtf8NoBom $rasPath)
  try { Remove-Item -LiteralPath $dslPath -Force -ErrorAction SilentlyContinue } catch { }
  try { Remove-Item -LiteralPath $rasPath -Force -ErrorAction SilentlyContinue } catch { }
  return $rasText
}

$result = [ordered]@{
  ok = $false
  rasText = ""
  aiAnalysis = ""
  aiSpec = ""
  summary = ""
  variants = @()
  err = ""
}

$script:statusPath = ""
function WriteStatus([string]$msg) {
  try {
    if ([string]::IsNullOrWhiteSpace($script:statusPath)) { return }
    $m = $msg
    if ($null -eq $m) { $m = "" }
    $m = $m.Trim()
    WriteUtf8NoBom $script:statusPath ($m + "`n")
  } catch { }
}

try {
  $inPath = $In
  $outPath = $Out
  if (-not [System.IO.Path]::IsPathRooted($inPath)) { $inPath = Join-Path -Path $PSScriptRoot -ChildPath $inPath }
  if (-not [System.IO.Path]::IsPathRooted($outPath)) { $outPath = Join-Path -Path $PSScriptRoot -ChildPath $outPath }

  if (-not (Test-Path -LiteralPath $inPath)) { throw "Input JSON not found: $In" }

  $inputs = (Get-Content -Raw -Encoding UTF8 -LiteralPath $inPath) | ConvertFrom-Json
  try {
    $sp = [string]$inputs.statusPath
    if (-not [string]::IsNullOrWhiteSpace($sp)) {
      if (-not [System.IO.Path]::IsPathRooted($sp)) { $sp = Join-Path -Path $PSScriptRoot -ChildPath $sp }
      $script:statusPath = $sp
    }
  } catch { $script:statusPath = "" }
  WriteStatus "start"
  $modeIndex = [int]$inputs.modeIndex
  $durationSec = 4.0
  [void][double]::TryParse([string]$inputs.durationText, [ref]$durationSec)
  if ($durationSec -lt 0.1) { throw "Duration too small." }

  $nameStem = SanitizeFileStem ([string]$inputs.nameStem)
  $rasText = ""
  $modeName = ""
  try { $modeName = [string]$inputs.modeName } catch { $modeName = "" }

  if ($modeIndex -eq 0) {
    $prompt = [string]$inputs.prompt
    $useOnline = [bool]$inputs.useOnline
    $freeKeys = $false
    try { $freeKeys = [bool]$inputs.freeKeys } catch { $freeKeys = $false }
    $variantsCount = 1
    try { $variantsCount = [int]$inputs.variantsCount } catch { $variantsCount = 1 }
    if ($variantsCount -lt 1) { $variantsCount = 1 }
    if ($variantsCount -gt 5) { $variantsCount = 5 }
    $preferLocal = $true
    try { $preferLocal = [bool]$inputs.preferLocal } catch { $preferLocal = $true }
    # Plugin mode: prefer real online AI behavior unless caller explicitly disables online.
    if ($useOnline) { $preferLocal = $false }
    $sketchTv = TryGetSketchTvFromInputs $inputs
    $sketchMode = GetSketchModeFromInputs $inputs
    $sketchTvForGen = $sketchTv
    if ($sketchMode -eq "reference" -or $sketchMode -eq "none") { $sketchTvForGen = $null }
    $prevCurveTv = TryGetPrevCurveTvFromInputs $inputs
    $selectedCurveTv = TryGetSelectedCurveTvFromInputs $inputs
    $chatText = TryGetChatTextFromInputs $inputs
    $rerollMode = ""
    try { $rerollMode = [string]$inputs.rerollMode } catch { $rerollMode = "" }
    $rerollMode = ($rerollMode.Trim().ToLowerInvariant())
    $revisionText = ""
    try { $revisionText = [string]$inputs.revisionText } catch { $revisionText = "" }
    $hasRev = (-not [string]::IsNullOrWhiteSpace($revisionText))
    $revLarge = $false
    if ($hasRev) { $revLarge = RevisionRequestsLargeReshape $revisionText }
    if ($hasRev -and ($null -eq $prevCurveTv -or $prevCurveTv.Count -lt 8) -and $null -ne $selectedCurveTv -and $selectedCurveTv.Count -ge 8) {
      # Revision should anchor on current selected curve when explicit prevCurve is not provided.
      $prevCurveTv = $selectedCurveTv
    }
    if ($hasRev) { $preferLocal = $false }
    if ([string]::IsNullOrWhiteSpace($prompt)) {
      # Allow sketch-only generation when online AI is enabled.
      if ($useOnline -and $null -ne $sketchTv -and $sketchTv.Count -ge 16) {
        $prompt = "" # keep empty so the model relies on sketch meaning
      } else {
        throw "Prompt is empty."
      }
    }

    # Local expert templates (more like "assistant-generated" results), only when no sketch is guiding the AI.
    try {
      if (-not $hasRev -and $preferLocal -and ($null -eq $sketchTv -or $sketchTv.Count -lt 2)) {
        if (PromptHasPulseIntent $prompt) {
          $bpmLocal = TryExtractBpmFromPrompt $prompt 75.0
          WriteStatus "local expert: heartbeat"
          $keys = MakeHeartbeatKeys $durationSec $bpmLocal
          $rasText = BuildRasCurveTextFromKeys ("heartbeat_pulse_" + [int][Math]::Round($bpmLocal) + "bpm") $keys
          try { $result.aiAnalysis = (CN @(0x672C,0x5730,0x4E13,0x5BB6,0xFF1A,0x5FC3,0x8DF3,0x8109,0x51B2,0x66F2,0x7EBF)) + " (BPM=" + ([int][Math]::Round($bpmLocal)) + ", 0-1)" } catch { }
          WriteStatus "done"
        }
      }
    } catch { }

    if (-not [string]::IsNullOrWhiteSpace($rasText)) {
      # Already handled by a local expert generator.
    } else {
      $useServerProxy = $false
      try { $useServerProxy = [bool]$inputs.useServerProxy } catch { $useServerProxy = $false }

      if ($useOnline -and $useServerProxy) {
        $serverUrl = ""
        $username = ""
        $deviceId = ""
        try { $serverUrl = [string]$inputs.serverUrl } catch { $serverUrl = "" }
        try { $username = [string]$inputs.username } catch { $username = "" }
        try { $deviceId = [string]$inputs.deviceId } catch { $deviceId = "" }

        # Prefer selectedCurve.tv, else prevCurve.tv; both are arrays of [t,v].
        $curveTv = $null
        try { $curveTv = $inputs.selectedCurve.tv } catch { $curveTv = $null }
        if ($null -eq $curveTv) {
          try { $curveTv = $inputs.prevCurve.tv } catch { $curveTv = $null }
        }

        $curveOut = @()
        if ($null -ne $curveTv -and ($curveTv -is [System.Array])) {
          foreach ($p in $curveTv) {
            try {
              if ($null -ne $p -and ($p -is [System.Array]) -and $p.Count -ge 2) {
                $curveOut += ,@([double]$p[0], [double]$p[1])
              }
            } catch { }
          }
        }

        WriteStatus "server proxy"
        $respSrv = InvokeRhinobirdServerCurve $serverUrl $username $deviceId $prompt $revisionText $durationSec $variantsCount $curveOut
        $okSrv = $false
        try { $okSrv = [bool]$respSrv.ok } catch { $okSrv = $false }
        if (-not $okSrv) {
          $msgSrv = ""
          try { $msgSrv = [string]$respSrv.message } catch { $msgSrv = "" }
          if ([string]::IsNullOrWhiteSpace($msgSrv)) { $msgSrv = "Server AI failed." }
          throw $msgSrv
        }

        $varsSrv = $null
        try { $varsSrv = $respSrv.variants } catch { $varsSrv = $null }
        if ($null -eq $varsSrv -or -not ($varsSrv -is [System.Array]) -or $varsSrv.Count -lt 1) {
          throw "Server AI returned no variants."
        }

        $outVars = New-Object System.Collections.Generic.List[object]
        foreach ($v in $varsSrv) {
          if ($null -eq $v) { continue }
          $nm = ""
          $an = ""
          $keys = $null
          try { $nm = [string]$v.name } catch { $nm = "" }
          try { $an = [string]$v.analysis_cn } catch { $an = "" }
          if ([string]::IsNullOrWhiteSpace($an)) { try { $an = [string]$v.analysis } catch { $an = "" } }
          try { $keys = $v.keys } catch { $keys = $null }
          if ($null -eq $keys -or -not ($keys -is [System.Array]) -or $keys.Count -lt 2) { continue }
          if ([string]::IsNullOrWhiteSpace($nm)) { $nm = "AI" }

          $rasV = BuildRasCurveTextFromKeys $nm $keys
          $outVars.Add([pscustomobject]@{ name = $nm; aiAnalysis = $an; rasText = $rasV }) | Out-Null
        }

        if ($outVars.Count -lt 1) {
          throw "Server AI returned no valid curve."
        }

        $result.variants = $outVars.ToArray()
        try { $result.aiAnalysis = [string]$result.variants[0].aiAnalysis } catch { $result.aiAnalysis = "" }
        $rasText = [string]$result.variants[0].rasText
        WriteStatus "done"
      } else {
      $apiKey = ""
      try { $apiKey = [string]$inputs.apiKey } catch { $apiKey = "" }
      if ([string]::IsNullOrWhiteSpace($apiKey)) { $apiKey = $env:AI_CURVE_TOOL_API_KEY }
      if ($useOnline -and [string]::IsNullOrWhiteSpace($apiKey)) { throw "Missing API key (OPENAI_API_KEY)." }

      if ($useOnline) {
      # Online pipeline:
      # - freeKeys: one-shot (analysis+keys together, more chat-like; better consistency)
      # - DSL mode: spec -> DSL -> rascurve (legacy)
      $baseUrl = [string]$inputs.baseUrl
      $model = [string]$inputs.model

      $basePrompt = $prompt
      $prompt2 = $prompt
      if ([string]::IsNullOrWhiteSpace($prompt2)) { $prompt2 = "(sketch-only; infer curve style)" }
      if ($hasRev) { $prompt2 = ($basePrompt.Trim() + "`n`nUserRevision:`n" + $revisionText.Trim()) }

      $fixHint = ""
      if ($hasRev) {
        if ($revLarge) {
          $fixHint = "Revision mode: user allows bigger reshape, but keep timing beats understandable and preserve continuity with previous curve."
        } else {
          $fixHint = "Revision mode: stay close to PreviousCurve and apply targeted local edits only. Do not rebuild into a totally different curve."
        }
      }

      # "Draw cards": generate multiple variants for the initial generation.
      if ($freeKeys -and (-not $hasRev) -and $variantsCount -gt 1) {
        # Keep style guidance open-ended so the model can follow natural conversation intent.
        $variantStylesText = @"
1) closely follow user intent (anchor option)
2) timing-focused reinterpretation
3) bold creative reinterpretation (still keep intent core)
"@.Trim()
        if ($variantsCount -le 2) {
          $variantStylesText = @"
1) closely follow user intent
2) clearly different reinterpretation (timing or energy)
"@.Trim()
        }

        $fixV = ""
        $outVars = $null
        for ($att = 0; $att -lt 2; $att++) {
          WriteStatus ("variants x{0}" -f $variantsCount)
          $objV = InvokeAiCurveVariantsOneShot $baseUrl $model $apiKey $prompt2 $variantsCount $variantStylesText $sketchTv $sketchMode $prevCurveTv $selectedCurveTv $fixV $chatText
          $arrV = $null
          try { $arrV = $objV.variants } catch { $arrV = $null }
          if ($null -eq $arrV -or -not ($arrV -is [System.Array]) -or $arrV.Count -ne $variantsCount) {
            $fixV = "Return JSON with variants: array of length VariantsCount. Each variant must include name, analysis_cn, keys."
            continue
          }

          $vars = New-Object System.Collections.Generic.List[object]
          $samples = New-Object System.Collections.Generic.List[object]
          $bad = ""
          for ($vi = 0; $vi -lt $arrV.Count; $vi++) {
            $oneV = $arrV[$vi]
            $analysisV = ""
            try { $analysisV = [string]$oneV.analysis_cn } catch { $analysisV = "" }
            $tvArrV = ExtractAiTvKeys $oneV
            $keysV = NormalizeAndSlopeKeys $tvArrV
            if ($null -eq $keysV -or $keysV.Count -lt 2) { $bad = ("variant {0} keys invalid" -f ($vi + 1)); break }

            # Enforce endpoints only if user explicitly asked.
            $pe2 = TryExtractStartEndValueFromPrompt $basePrompt
            if ($pe2.hasStart) { try { $keysV[0].t = 0.0; $keysV[0].v = [double](Clamp ([double]$pe2.start) 0.0 1.0) } catch { } }
            if ($pe2.hasEnd) { try { $keysV[$keysV.Count - 1].t = 1.0; $keysV[$keysV.Count - 1].v = [double](Clamp ([double]$pe2.end) 0.0 1.0) } catch { } }
            if ((PromptHasPulseIntent $basePrompt) -and -not $pe2.hasStart -and -not $pe2.hasEnd) {
              try { $keysV[0].t = 0.0; $keysV[0].v = 0.0 } catch { }
              try { $keysV[$keysV.Count - 1].t = 1.0; $keysV[$keysV.Count - 1].v = 0.0 } catch { }
            }

            $nameV = "AI_V" + ($vi + 1)
            try { if (-not [string]::IsNullOrWhiteSpace([string]$oneV.name)) { $nameV = ("AI_" + ([string]$oneV.name).Trim()) } } catch { }
            $rasV = BuildRasCurveTextFromKeys $nameV $keysV

            # Quick validation (lightweight) for major intent mismatches.
            $keysFullV = ParseRasCurveFullKeysFromText $rasV
            $valsV = SampleCurveValues $keysFullV 160
            $anS = AnalyzeSampled $valsV
            try { $samples.Add($valsV) | Out-Null } catch { }

            $vars.Add([pscustomobject]@{
              name = $nameV
              aiAnalysis = $analysisV.Trim()
              aiSpec = ($oneV | ConvertTo-Json -Depth 8)
              rasText = $rasV
            }) | Out-Null
          }

          if (-not [string]::IsNullOrWhiteSpace($bad)) {
            $fixV = ("One or more variants are invalid: {0}. Fix them and keep all values in 0..1." -f $bad)
            continue
          }

          # Encourage meaningful diversity between cards.
          # Diversity is a preference; do not force extra retries.

          $outVars = $vars.ToArray()
          break
        }

        if ($null -eq $outVars -or $outVars.Count -lt 1) { throw "Failed to generate variants." }
        $result.variants = $outVars
        try { $result.rasText = [string]$result.variants[0].rasText } catch { }
        try { $result.aiAnalysis = [string]$result.variants[0].aiAnalysis } catch { }
        try { $result.aiSpec = [string]$result.variants[0].aiSpec } catch { }
        $rasText = [string]$result.rasText
      } else {
        $maxAttempts = 2
        for ($attempt = 0; $attempt -lt $maxAttempts; $attempt++) {
          if ($freeKeys) {
            WriteStatus ("step1/1: keys" + $(if ($attempt -gt 0) { " (fix)" } else { "" }))

            $one = InvokeAiCurveOneShot $baseUrl $model $apiKey $prompt2 $sketchTv $sketchMode $prevCurveTv $fixHint $chatText
            $analysisCn = ""
            try { $analysisCn = [string]$one.analysis_cn } catch { $analysisCn = "" }
            try { $result.aiAnalysis = $analysisCn.Trim() } catch { $result.aiAnalysis = "" }
            try { $result.aiSpec = ($one | ConvertTo-Json -Depth 8) } catch { $result.aiSpec = "" }

            $tvArr = ExtractAiTvKeys $one
            $keysK = NormalizeAndSlopeKeys $tvArr
            if ($null -eq $keysK -or $keysK.Count -lt 2) {
              $fixHint = "Return JSON with keys: [{t:0..1,v:0..1}, ...] (4-40 points)."
              if ($attempt -lt ($maxAttempts - 1)) { continue }
              throw "AI keys invalid."
            }

            # Enforce endpoints only if user explicitly asked.
            $pe2 = TryExtractStartEndValueFromPrompt $basePrompt
            if ($pe2.hasStart) { try { $keysK[0].t = 0.0; $keysK[0].v = [double](Clamp ([double]$pe2.start) 0.0 1.0) } catch { } }
            if ($pe2.hasEnd) { try { $keysK[$keysK.Count - 1].t = 1.0; $keysK[$keysK.Count - 1].v = [double](Clamp ([double]$pe2.end) 0.0 1.0) } catch { } }
            if ((PromptHasPulseIntent $basePrompt) -and -not $pe2.hasStart -and -not $pe2.hasEnd) {
              try { $keysK[0].t = 0.0; $keysK[0].v = 0.0 } catch { }
              try { $keysK[$keysK.Count - 1].t = 1.0; $keysK[$keysK.Count - 1].v = 0.0 } catch { }
            }

            if ($hasRev -and $null -ne $prevCurveTv -and $prevCurveTv.Count -ge 8) {
              $blend = if ($revLarge) { 0.72 } else { 0.38 }
              $maxDelta = if ($revLarge) { 0.40 } else { 0.22 }
              $keysK = ConstrainRevisionKeysToPrevious $keysK $prevCurveTv $blend $maxDelta
            }

            $nameK = "AI_Free"
            try { if (-not [string]::IsNullOrWhiteSpace([string]$one.name)) { $nameK = ("AI_" + ([string]$one.name).Trim()) } } catch { }
            $rasText = BuildRasCurveTextFromKeys $nameK $keysK
          } else {
          # Legacy DSL path (kept for compatibility).
          WriteStatus ("step1/2: spec" + $(if ($attempt -gt 0) { " (fix)" } else { "" }))

          $prevSpecAll = $null
          try {
            $prevJson = [string]$inputs.aiSpecPrev
            if (-not [string]::IsNullOrWhiteSpace($prevJson)) { $prevSpecAll = ($prevJson | ConvertFrom-Json) }
          } catch { $prevSpecAll = $null }

          if ($hasRev -and $null -ne $prevSpecAll) {
            $specAll = InvokeAiCurveSpecRevision $baseUrl $model $apiKey $basePrompt $revisionText $prevSpecAll $prevCurveTv $sketchTv
          } else {
            $specAll = InvokeAiCurveSpec $baseUrl $model $apiKey $prompt2 $sketchTv
          }
          $analysisCn = ""
          try { $analysisCn = [string]$specAll.analysis_cn } catch { $analysisCn = "" }
          $specObj = $null
          try { $specObj = $specAll.spec } catch { $specObj = $null }
          if ($null -eq $specObj) { $specObj = [pscustomobject]@{} }
          try { $result.aiAnalysis = $analysisCn.Trim() } catch { $result.aiAnalysis = "" }
          try { $result.aiSpec = ($specAll | ConvertTo-Json -Depth 8) } catch { $result.aiSpec = "" }

          WriteStatus ("step2/2: dsl" + $(if ($attempt -gt 0) { " (fix)" } else { "" }))

          $rootDir = [System.IO.Path]::GetFullPath((Join-Path -Path $PSScriptRoot -ChildPath ".."))
          $pluginDir = Join-Path -Path $rootDir -ChildPath (CN @(0x7280,0x725B,0x52A8,0x753B,0x63D2,0x4EF6))
          $aiDir = Join-Path -Path $pluginDir -ChildPath "AI"
          $toolDslToRas = Join-Path -Path $aiDir -ChildPath "dsl_to_rascurve.ps1"
          if (-not (Test-Path -LiteralPath $toolDslToRas)) { throw "Missing tool: $toolDslToRas" }

          $dsl = InvokeAiDslFromPromptAndSpec $baseUrl $model $apiKey $prompt2 $specAll $sketchTvForGen $fixHint

          # Pre-validate/fix: stack DSL requires a 'base' object.
          $expectBounce0 = $false
          try { $expectBounce0 = [bool]$specObj.expect_bounce } catch { $expectBounce0 = $false }
          if (-not $expectBounce0) { $expectBounce0 = PromptHasBounceIntent $prompt }
          $dsl = EnsureStackDslHasBase $dsl $expectBounce0
          $dsl = EnsureDslEndpoints $dsl $specAll $prompt $sketchTv
          if (-not (HasProp $dsl "base") -or $null -eq $dsl.base) {
            $fixHint = "Your DSL is invalid: stack DSL requires a 'base' object. Return a valid stack DSL with base.model set."
            if ($attempt -lt ($maxAttempts - 1)) { continue }
          }

          WriteStatus ("render rascurve" + $(if ($attempt -gt 0) { " (fix)" } else { "" }))
          $rasText = WriteTempDslAndGenerateRas $toolDslToRas $dsl
          }

        # Shared validation for both pipelines.
        WriteStatus "validate"
        $tmpDirV = Join-Path -Path $PSScriptRoot -ChildPath "out"
        if (-not (Test-Path -LiteralPath $tmpDirV)) { New-Item -ItemType Directory -Force -Path $tmpDirV | Out-Null }
        $tmpRasV = Join-Path -Path $tmpDirV -ChildPath ("_validate_" + [Guid]::NewGuid().ToString("N") + ".rascurve")
        WriteUtf8NoBom $tmpRasV $rasText
        $keysTv = ParseRasCurveTV $tmpRasV
        try { Remove-Item -LiteralPath $tmpRasV -Force -ErrorAction SilentlyContinue } catch { }
        $an = AnalyzeKeysBasic $keysTv
        $keysFullV = ParseRasCurveFullKeysFromText $rasText
        $valsV = SampleCurveValues $keysFullV 160
        $anS = AnalyzeSampled $valsV

        # Loosen bounce enforcement: treat "bounce/spring" as preference, not a hard requirement.
        # We still keep severe checks (e.g. invalid keys, tiny amplitude) and sketch-fit when sketch is a hard constraint.

        # Keep auto-fix light in free-keys mode to avoid over-constraining model creativity.
        if (-not $freeKeys) {
          # Range sanity (normalized 0..1).
          $specMin = 0.0
          $specMax = 1.0
          if ($anS.ok) {
            $desiredRange = [Math]::Abs([double]$specMax - [double]$specMin)
            $actualRange = [Math]::Abs([double]$anS.maxV - [double]$anS.minV)
            # Only auto-fix for severe amplitude mismatch.
            if ($desiredRange -ge 0.90 -and $actualRange -lt 0.25) {
              $fixHint = ("The curve amplitude is far too small. Spec value range is [{0:F2},{1:F2}] but sampled output range is [{2:F3},{3:F3}]. Scale up so the curve spans most of the target range. Prefer amplitude_value (absolute 0..1) for pulse/bounce layers; do NOT shrink everything via a small end_value-start_value range. Keep values within 0..1, key_style=extrema, max_keys 6..12." -f $specMin, $specMax, $anS.minV, $anS.maxV)
              if ($attempt -lt ($maxAttempts - 1)) { continue }
            }
          }

          # Heartbeat/pulse: ensure it actually looks like a pulse (baseline near 0, strong main spike).
          $expectPulse = PromptHasPulseIntent $prompt
          if ($expectPulse -and $anS.ok) {
            if ($anS.maxV -lt 0.75) {
              $fixHint = ("Heartbeat/pulse should have a strong main spike near 1.0. Current max={0:F3}. Increase the peak amplitude." -f $anS.maxV)
              if ($attempt -lt ($maxAttempts - 1)) { continue }
            }
          }

          if ($sketchMode -eq "constraint" -and $null -ne $sketchTv -and $sketchTv.Count -ge 16) {
            $fit = ComputeSketchFit $keysFullV $sketchTv
            if ($fit.ok -and ($fit.mae -gt 0.10 -or $fit.maxe -gt 0.22)) {
              $fixHint = ("Match the sketch more closely. Current error: MAE={0:F3}, MAX={1:F3}. Reduce error while keeping values in 0..1." -f $fit.mae, $fit.maxe)
              if ($attempt -lt ($maxAttempts - 1)) { continue }
            }
          }
        }

          break
        }
      }
      WriteStatus "done"
      } else {
      # Local-only fallback: avoid DSL tools (they are fragile and can fail on regex/stack-base issues).
      WriteStatus "local fallback"
      $pe = TryExtractStartEndValueFromPrompt $prompt
      $svL = if ($pe.hasStart) { [double]$pe.start } else { 0.0 }
      $evL = if ($pe.hasEnd) { [double]$pe.end } else { 1.0 }
      if (PromptHasPulseIntent $prompt) { $svL = 0.0; $evL = 0.0 }

      $keysL = $null
      if (PromptHasPulseIntent $prompt) {
        $bpmLocal2 = TryExtractBpmFromPrompt $prompt 75.0
        $keysL = MakeHeartbeatKeys $durationSec $bpmLocal2
        try { $result.aiAnalysis = (CN @(0x672C,0x5730,0x751F,0x6210,0xFF1A)) + (CN @(0x5FC3,0x8DF3,0x8109,0x51B2,0x0020,0x66F2,0x7EBF)) } catch { }
      } elseif (PromptHasBounceIntent $prompt) {
        $keysL = MakeLocalInRangeSpringKeys $svL $evL $prompt
        try { $result.aiAnalysis = (CN @(0x672C,0x5730,0x751F,0x6210,0xFF1A)) + (CN @(0x786C,0x5F39,0x7C27,0x0020,0x0028,0x5C11,0x56DE,0x5F39,0x0029)) } catch { }
      } else {
        $keysL = MakeLocalEaseKeys $svL $evL $prompt
        try { $result.aiAnalysis = (CN @(0x672C,0x5730,0x751F,0x6210,0xFF1A)) + (CN @(0x5E73,0x6ED1,0x7F13,0x52A8)) } catch { }
      }

      if ($null -eq $keysL -or $keysL.Count -lt 2) { throw "Local fallback failed." }
      $rasText = BuildRasCurveTextFromKeys ("local_" + $nameStem) $keysL
      }
    }
    }
  } elseif ($modeIndex -eq 1) {
    WriteStatus "preset: spring"
    $center = 0.5
    [void][double]::TryParse([string]$inputs.centerText, [ref]$center)
    $center = Clamp $center 0.0 1.0
    $keys = MakeHardSpringFrontloadedKeys $durationSec $center 0.48 0.25 3.0 0.01 10.0
    $rasText = BuildRasCurveTextFromKeys ("hard_spring_frontloaded_center_" + $center) $keys
  } else {
    WriteStatus "preset: heartbeat"
    $bpm = 75.0
    [void][double]::TryParse([string]$inputs.bpmText, [ref]$bpm)
    $keys = MakeHeartbeatKeys $durationSec $bpm
    $rasText = BuildRasCurveTextFromKeys ("heartbeat_" + [int][Math]::Round($bpm) + "bpm") $keys
  }

  # Optional AI summary when online is enabled.
  try {
    $useOnline2 = $false
    try { $useOnline2 = [bool]$inputs.useOnline } catch { $useOnline2 = $false }
    $wantSum = $false
    try { $wantSum = [bool]$inputs.wantSummary } catch { $wantSum = $false }
    $apiKey2 = ""
    try { $apiKey2 = [string]$inputs.apiKey } catch { $apiKey2 = "" }
    if ($useOnline2 -and [string]::IsNullOrWhiteSpace($apiKey2)) { $apiKey2 = $env:AI_CURVE_TOOL_API_KEY }
    if ($useOnline2 -and $wantSum -and -not [string]::IsNullOrWhiteSpace($apiKey2) -and -not [string]::IsNullOrWhiteSpace($rasText)) {
      $tmpDir2 = Join-Path -Path $PSScriptRoot -ChildPath "out"
      if (-not (Test-Path -LiteralPath $tmpDir2)) { New-Item -ItemType Directory -Force -Path $tmpDir2 | Out-Null }
      $tmpRas2 = Join-Path -Path $tmpDir2 -ChildPath ("_summary_" + [Guid]::NewGuid().ToString("N") + ".rascurve")
      WriteUtf8NoBom $tmpRas2 $rasText
      $keysTv = ParseRasCurveTV $tmpRas2
      $stats = ComputeStats $keysTv
      $prompt0 = ""
      try { $prompt0 = [string]$inputs.prompt } catch { $prompt0 = "" }
      if ([string]::IsNullOrWhiteSpace($prompt0)) { $prompt0 = $modeName }
      $sum = GenerateAiSummary ([string]$inputs.baseUrl) ([string]$inputs.model) $apiKey2 $modeName $prompt0 $tmpRas2 $keysTv $stats
      if (-not [string]::IsNullOrWhiteSpace($sum)) { $result.summary = $sum }
      try { Remove-Item -LiteralPath $tmpRas2 -Force -ErrorAction SilentlyContinue } catch { }
    }
  } catch {
    # Ignore summary failures; generation still succeeds.
  }

  $result.ok = $true
  $result.rasText = [string]$rasText
  $result.summary = [string]$result.summary
} catch {
  $result.ok = $false
  $result.err = $_.Exception.Message
  WriteStatus ("error: " + $result.err)
}

WriteUtf8NoBom $Out (([pscustomobject]$result | ConvertTo-Json -Depth 6) + "`n")
